/*---------------------------------------------------------------------------------
 * 3D on top (spinning cuboid). Bottom = scripting editor: play/step/stop,
 * script lines (tokens), nav (up/down/clear), token toolbox. Script is
 * interpreted when play is clicked; step when paused; stop resets.
 *--------------------------------------------------------------------------------*/

#include <nds.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

/* Cube color options: red, green, blue (RGB15) */
static const u16 CUBE_COLORS[3] = {
    RGB15(31, 16, 16), /* pale red */
    RGB15(16, 31, 16), /* pale green */
    RGB15(16, 16, 31), /* pale blue */
};

/* Bottom screen bitmap (256x256 u16, we draw in 256x192 visible area) */
#define SUB_BMP_WIDTH 256
#define SUB_BMP_HEIGHT 256
#define SUB_VIS_HEIGHT 192

/* Software-rendered colors (RGB555, bit 15 = opaque) */
#define COLOR_BG (RGB15(8, 8, 10) | BIT(15))
#define COLOR_BTN (RGB15(20, 20, 24) | BIT(15))
#define COLOR_BTN_EDGE (RGB15(28, 28, 31) | BIT(15))
#define COLOR_TEXT (RGB15(31, 31, 31) | BIT(15))
#define COLOR_HIGHLIGHT (RGB15(14, 14, 18) | BIT(15))
#define COLOR_HIGHLIGHT_PAUSED (RGB15(14, 14, 18) | BIT(15)) /* slightly brighter than BG: cursor when paused */
#define COLOR_PLAYING (RGB15(24, 8, 8) | BIT(15))            /* red tint for stop button when script active (running or paused) */
#define COLOR_ARROW (RGB15(31, 31, 31) | BIT(15))

/* Script: list of token lines */
#define MAX_SCRIPT_LINES 128
#define MAX_TOKEN_LEN 16
static char script[MAX_SCRIPT_LINES][MAX_TOKEN_LEN];
static float scriptParam[MAX_SCRIPT_LINES];    /* literal number when scriptParamIsReg=0 */
static int scriptReg[MAX_SCRIPT_LINES];        /* 0-25 = A-Z for SET/ADD/SUBTRACT first param */
static int scriptParamIsReg[MAX_SCRIPT_LINES]; /* 1 = number param is from register (A-Z or extended) */
static int scriptParamReg[MAX_SCRIPT_LINES];   /* 0-34 when scriptParamIsReg=1 */
static float scriptParam2[MAX_SCRIPT_LINES];   /* second number param (IF_GT/IF_LT, TRANSLATE Y) */
static int scriptParamIsReg2[MAX_SCRIPT_LINES];
static int scriptParamReg2[MAX_SCRIPT_LINES];
static float scriptParam3[MAX_SCRIPT_LINES]; /* third number param (TRANSLATE Y) */
static int scriptParamIsReg3[MAX_SCRIPT_LINES];
static int scriptParamReg3[MAX_SCRIPT_LINES];
static float scriptParam4[MAX_SCRIPT_LINES]; /* fourth number param (TRANSLATE Z) */
static int scriptParamIsReg4[MAX_SCRIPT_LINES];
static int scriptParamReg4[MAX_SCRIPT_LINES];
static int scriptLines;
static int selectedLine;

#define NUM_SCRIPT_SLOTS 6
static int currentScriptSlot; /* 0..5 = which slot we're editing (script 1..6) */
static int scriptSlotLines[NUM_SCRIPT_SLOTS];
static char scriptSlot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES][MAX_TOKEN_LEN];
static float scriptParamSlot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static int scriptRegSlot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static int scriptParamIsRegSlot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static int scriptParamRegSlot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static float scriptParam2Slot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static int scriptParamIsReg2Slot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static int scriptParamReg2Slot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static float scriptParam3Slot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static int scriptParamIsReg3Slot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static int scriptParamReg3Slot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static float scriptParam4Slot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static int scriptParamIsReg4Slot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];
static int scriptParamReg4Slot[NUM_SCRIPT_SLOTS][MAX_SCRIPT_LINES];

/* Per-model state: only models added with MODEL <index> are rendered. */
#define MAX_MODELS 16
static int modelActive[MAX_MODELS];
static float modelAngle[MAX_MODELS];
static float modelX[MAX_MODELS], modelY[MAX_MODELS], modelZ[MAX_MODELS];
static int modelColorIndex[MAX_MODELS];

/* Registers A-Z (0-25) + read-only: Lft, Up, Rgt, Dn, kA, kB, Time, LOOKX, LOOKZ (26-34) */
#define NUM_REGISTERS 26
#define NUM_EXTENDED_REGISTERS 35
#define REG_LFT 26
#define REG_UP 27
#define REG_RGT 28
#define REG_DN 29
#define REG_BTN_A 30
#define REG_BTN_B 31
#define REG_TIME 32
#define REG_LOOKX 33
#define REG_LOOKZ 34
static float registers[NUM_REGISTERS];

/* Elapsed seconds since run started; advances only when running, resets on stop. */
static float elapsedTimeSeconds;

/* Pong: A=ball X, B=velocity, C=paddle Z, D=miss, E=hit, F=game-over countdown, G=beeps left. On miss: stop move, beep 3x, then reset. Miss detection only when F=0 so F/G count down. */
static const char *defaultScript[] = {
    "model", "model", "next_color", "next_color", "cam_pos", "cam_angle", "background", "set", "position", "set", "set", "loop",
    "if_lt", "add", "end_if",
    "if_gt", "set", "multiply", "beep", "end_if",
    "if_lt", "set", "if_lt", "if_gt", "set", "end_if", "if_lt", "set", "end_if", "if_true", "set", "set", "end_if",
    "set", "subtract", "if_true", "set", "multiply", "beep", "end_if", "end_if", "end_if",
    "if_gt", "subtract", "if_true", "beep", "subtract", "end_if", "end_if",
    "if_lt", "if_true", "set", "set", "set", "end_if", "end_if",
    "if_lt", "if_true", "next_color", "end_if", "if_true", "add", "end_if", "if_true", "add", "end_if", "end_if",
    "position", "position", "sleep", "end_loop"};
static const int defaultScriptLen = 71;

/* Numpad for editing number param (only when script not running) */
static int numpadActive;
static int numpadMode; /* 0 = numbers (default), 1 = register list A-Z + extended */
static int editingParamLine;
static int editingParamIndex; /* 0 = first param, 1 = second param (IF_GT/IF_LT) */
#define NUMPAD_BUF_LEN 16
static char numpadBuffer[NUMPAD_BUF_LEN];

/* Register selector popup for SET/ADD/SUBTRACT first param */
static int registerSelectActive;

/* Token view: 1 = showing token picker (replaces toolbox) */
static int tokenViewActive;

/* Script menu: 1 = showing script 1..6 picker */
static int scriptMenuActive;

/* Skip next script-area touch so it doesn't overwrite selectedLine after inserting from token picker */
static int skipNextScriptTouchAfterTokenInsert;

/* Execution state */
#define LOOP_STACK_MAX 4
static int scriptRunning; /* 1 = playing, 0 = paused or stopped */
static int scriptActive;  /* 1 = play was pressed and stop not pressed yet (running or paused); 0 = stopped. Used to disable token/clear and show red stop button */
static int scriptIP;      /* current line index */
static int loopStack[LOOP_STACK_MAX];
static int loopSp;
static int sleepFramesLeft; /* frames until sleep ends (~60 = 1 sec) */

static int scriptScrollOffset; /* first visible line index (for drawing) */

#define SCROLL_REPEAT_DELAY 20   /* frames before first repeat (~0.33s) */
#define SCROLL_REPEAT_INTERVAL 5 /* frames between repeats */
static int scrollUpCounter;      /* -1 = not holding, else countdown to next scroll */
static int scrollDownCounter;
static int stepRepeatCounter;  /* -1 = not holding Step, else countdown to next step (when paused) */
static int clearRepeatCounter; /* -1 = not holding Clear, else countdown to next clear */

/* Beep: 0.1s pong-style sound when BEEP token runs */
static int beepChannel;
static int beepFramesLeft; /* frames until we stop the beep (~6 = 0.1s at 60fps) */

/* Camera: set by CAM_POS and CAM_ANGLE (default: position 0,0,4; yaw 0, pitch 0) */
static float camX, camY, camZ;
static float camYaw, camPitch; /* degrees */

/* Top screen background: 0=black, 1=dark pale red, 2=dark pale green, 3=dark pale blue */
static int bgColorIndex;

static u16 *subBuffer;
static int subBgId;

/*---------------------------------------------------------------------------------
 * 5x7 bitmap font: A-Z (26), _ (1), 0-9 (10), . - +. 7 rows x 5 bits per glyph.
 *--------------------------------------------------------------------------------*/
#define FONT_W 5
#define FONT_H 7
static const u8 font_5x7[40][7] = {
    {0x0E, 0x11, 0x11, 0x1F, 0x11, 0x11, 0x11}, /* A */
    {0x1E, 0x11, 0x11, 0x1E, 0x11, 0x11, 0x1E}, /* B */
    {0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F}, /* C */
    {0x1E, 0x11, 0x11, 0x11, 0x11, 0x11, 0x1E}, /* D */
    {0x1F, 0x10, 0x10, 0x1E, 0x10, 0x10, 0x1F}, /* E */
    {0x1F, 0x10, 0x10, 0x1E, 0x10, 0x10, 0x10}, /* F */
    {0x0F, 0x10, 0x10, 0x17, 0x11, 0x11, 0x0E}, /* G */
    {0x11, 0x11, 0x11, 0x1F, 0x11, 0x11, 0x11}, /* H */
    {0x0E, 0x04, 0x04, 0x04, 0x04, 0x04, 0x0E}, /* I */
    {0x07, 0x02, 0x02, 0x02, 0x02, 0x12, 0x0C}, /* J */
    {0x11, 0x12, 0x14, 0x18, 0x14, 0x12, 0x11}, /* K */
    {0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x1F}, /* L */
    {0x11, 0x1B, 0x15, 0x15, 0x11, 0x11, 0x11}, /* M */
    {0x11, 0x19, 0x15, 0x13, 0x11, 0x11, 0x11}, /* N */
    {0x0E, 0x11, 0x11, 0x11, 0x11, 0x11, 0x0E}, /* O */
    {0x1E, 0x11, 0x11, 0x1E, 0x10, 0x10, 0x10}, /* P */
    {0x0E, 0x11, 0x11, 0x11, 0x15, 0x12, 0x0D}, /* Q */
    {0x1E, 0x11, 0x11, 0x1E, 0x14, 0x12, 0x11}, /* R */
    {0x0F, 0x10, 0x10, 0x0E, 0x01, 0x01, 0x1E}, /* S */
    {0x1F, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04}, /* T */
    {0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x0E}, /* U */
    {0x11, 0x11, 0x11, 0x11, 0x0A, 0x0A, 0x04}, /* V */
    {0x11, 0x11, 0x11, 0x15, 0x15, 0x1B, 0x11}, /* W */
    {0x11, 0x11, 0x0A, 0x04, 0x0A, 0x11, 0x11}, /* X */
    {0x11, 0x11, 0x0A, 0x04, 0x04, 0x04, 0x04}, /* Y */
    {0x1F, 0x01, 0x02, 0x04, 0x08, 0x10, 0x1F}, /* Z */
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1F}, /* _ */
    {0x0E, 0x11, 0x13, 0x15, 0x19, 0x11, 0x0E}, /* 0 */
    {0x04, 0x0C, 0x04, 0x04, 0x04, 0x04, 0x0E}, /* 1 */
    {0x0E, 0x11, 0x01, 0x02, 0x04, 0x08, 0x1F}, /* 2 */
    {0x1F, 0x02, 0x04, 0x02, 0x01, 0x11, 0x0E}, /* 3 */
    {0x02, 0x06, 0x0A, 0x12, 0x1F, 0x02, 0x02}, /* 4 */
    {0x1F, 0x10, 0x1E, 0x01, 0x01, 0x11, 0x0E}, /* 5 */
    {0x06, 0x08, 0x10, 0x1E, 0x11, 0x11, 0x0E}, /* 6 */
    {0x1F, 0x01, 0x02, 0x04, 0x08, 0x08, 0x08}, /* 7 */
    {0x0E, 0x11, 0x11, 0x0E, 0x11, 0x11, 0x0E}, /* 8 */
    {0x0E, 0x11, 0x11, 0x0F, 0x01, 0x02, 0x0C}, /* 9 */
    {0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00}, /* . */
    {0x00, 0x00, 0x00, 0x1F, 0x00, 0x00, 0x00}, /* - */
    {0x04, 0x04, 0x04, 0x1F, 0x04, 0x04, 0x04}, /* + */
};

static void drawChar(int x, int y, char c, u16 color)
{
    int idx, row, col;
    if (c >= 'a' && c <= 'z')
        c -= 32;
    if (c >= 'A' && c <= 'Z')
        idx = c - 'A';
    else if (c == '_')
        idx = 26;
    else if (c >= '0' && c <= '9')
        idx = 27 + (c - '0');
    else if (c == '.')
    {
        /* Period: small dot at baseline (2x2), centered in cell */
        int pw = 2, ph = 2;
        int px0 = x + (FONT_W - pw) / 2;
        int py0 = y + FONT_H - ph;
        int row, col;
        if (px0 + pw > SUB_BMP_WIDTH || py0 + ph > SUB_BMP_HEIGHT)
            return;
        for (row = 0; row < ph; row++)
        {
            int sy = py0 + row;
            if (sy < 0 || sy >= SUB_BMP_HEIGHT)
                continue;
            for (col = 0; col < pw; col++)
            {
                int sx = px0 + col;
                if (sx >= 0 && sx < SUB_BMP_WIDTH)
                    subBuffer[sy * SUB_BMP_WIDTH + sx] = color;
            }
        }
        return;
    }
    else if (c == '-')
        idx = 38;
    else if (c == '+')
        idx = 39;
    else
        return; /* space or unsupported: skip */
    if (x + FONT_W > SUB_BMP_WIDTH || y + FONT_H > SUB_BMP_HEIGHT)
        return;
    for (row = 0; row < FONT_H; row++)
    {
        u8 bits = font_5x7[idx][row];
        for (col = 0; col < FONT_W; col++)
        {
            if (bits & (1 << col))
                subBuffer[(y + row) * SUB_BMP_WIDTH + (x + (FONT_W - 1 - col))] = color;
        }
    }
}

static void drawString(int x, int y, const char *str, u16 color)
{
    while (*str)
    {
        if (*str != ' ')
            drawChar(x, y, *str, color);
        x += FONT_W + 1;
        str++;
    }
}

/* Pixel width of string (for centering). */
static int stringWidth(const char *str)
{
    int n = 0;
    while (*str)
    {
        n++;
        str++;
    }
    return n > 0 ? (n * (FONT_W + 1)) - 1 : 0;
}

/* Draw string centered in box (x0,y0)-(x1,y1). */
static void drawStringCentered(int x0, int y0, int x1, int y1, const char *str, u16 color)
{
    int w = stringWidth(str);
    int h = FONT_H;
    int x = (x0 + x1) / 2 - w / 2;
    int y = (y0 + y1) / 2 - h / 2;
    drawString(x, y, str, color);
}

/* Read-only value for extended reg index 8-14 (keys, time). */
static float getExtendedRegisterValue(int regIndex)
{
    u16 keys = keysHeld();
    switch (regIndex)
    {
    case REG_LFT:
        return (keys & KEY_LEFT) ? 1.0f : 0.0f;
    case REG_UP:
        return (keys & KEY_UP) ? 1.0f : 0.0f;
    case REG_RGT:
        return (keys & KEY_RIGHT) ? 1.0f : 0.0f;
    case REG_DN:
        return (keys & KEY_DOWN) ? 1.0f : 0.0f;
    case REG_BTN_A:
        return (keys & KEY_A) ? 1.0f : 0.0f;
    case REG_BTN_B:
        return (keys & KEY_B) ? 1.0f : 0.0f;
    case REG_TIME:
        return elapsedTimeSeconds;
    case REG_LOOKX:
    {
        float yaw_rad = camYaw * (3.14159265f / 180.0f);
        return sinf(yaw_rad); /* normalized horizontal look X (Y up) */
    }
    case REG_LOOKZ:
    {
        float yaw_rad = camYaw * (3.14159265f / 180.0f);
        return -cosf(yaw_rad); /* normalized horizontal look Z */
    }
    default:
        return 0.0f;
    }
}

/* Get number param value for line (from register or literal). paramIndex: 0 = first, 1 = second, 2 = third, 3 = fourth (TRANSLATE Z). Reg 0-25 = A-Z, 26-34 = read-only Lft/Up/.../LOOKZ. */
static float getNumberParamValue(int lineIdx, int paramIndex)
{
    int r;
    if (lineIdx < 0 || lineIdx >= scriptLines)
        return 0.0f;
    if (paramIndex == 1)
    {
        if (!scriptParamIsReg2[lineIdx])
            return scriptParam2[lineIdx];
        r = scriptParamReg2[lineIdx];
        if (r >= 0 && r < NUM_REGISTERS)
            return registers[r];
        if (r >= REG_LFT && r < NUM_EXTENDED_REGISTERS)
            return getExtendedRegisterValue(r);
        return scriptParam2[lineIdx];
    }
    if (paramIndex == 2)
    {
        if (!scriptParamIsReg3[lineIdx])
            return scriptParam3[lineIdx];
        r = scriptParamReg3[lineIdx];
        if (r >= 0 && r < NUM_REGISTERS)
            return registers[r];
        if (r >= REG_LFT && r < NUM_EXTENDED_REGISTERS)
            return getExtendedRegisterValue(r);
        return scriptParam3[lineIdx];
    }
    if (paramIndex == 3)
    {
        if (!scriptParamIsReg4[lineIdx])
            return scriptParam4[lineIdx];
        r = scriptParamReg4[lineIdx];
        if (r >= 0 && r < NUM_REGISTERS)
            return registers[r];
        if (r >= REG_LFT && r < NUM_EXTENDED_REGISTERS)
            return getExtendedRegisterValue(r);
        return scriptParam4[lineIdx];
    }
    if (scriptParamIsReg[lineIdx])
    {
        r = scriptParamReg[lineIdx];
        if (r >= 0 && r < NUM_REGISTERS)
            return registers[r];
        if (r >= REG_LFT && r < NUM_EXTENDED_REGISTERS)
            return getExtendedRegisterValue(r);
    }
    return scriptParam[lineIdx];
}

/* Label for reg index 0-34 (A-Z, Lft, Up, Rgt, Dn, kA, kB, Time, LOOKX, LOOKZ). */
static void getRegisterLabel(int regIndex, char *out, int maxLen)
{
    if (regIndex >= 0 && regIndex < NUM_REGISTERS)
    {
        out[0] = 'A' + (char)regIndex;
        out[1] = '\0';
        return;
    }
    switch (regIndex)
    {
    case REG_LFT:
        snprintf(out, (unsigned)maxLen, "Lft");
        break;
    case REG_UP:
        snprintf(out, (unsigned)maxLen, "Up");
        break;
    case REG_RGT:
        snprintf(out, (unsigned)maxLen, "Rgt");
        break;
    case REG_DN:
        snprintf(out, (unsigned)maxLen, "Dn");
        break;
    case REG_BTN_A:
        snprintf(out, (unsigned)maxLen, "kA");
        break;
    case REG_BTN_B:
        snprintf(out, (unsigned)maxLen, "kB");
        break;
    case REG_TIME:
        snprintf(out, (unsigned)maxLen, "Time");
        break;
    case REG_LOOKX:
        snprintf(out, (unsigned)maxLen, "LOOKX");
        break;
    case REG_LOOKZ:
        snprintf(out, (unsigned)maxLen, "LOOKZ");
        break;
    default:
        out[0] = '?';
        out[1] = '\0';
        break;
    }
}

static void drawRegisterAt(int x, int y, int regIndex, u16 color)
{
    char buf[8];
    getRegisterLabel(regIndex, buf, sizeof(buf));
    drawString(x, y, buf, color);
}

/* Format number param for display (preserve decimals and sign e.g. "1.0001", "-1.5"). */
static void formatSleepParam(float v, char *out, int maxLen)
{
    int n;
    if (v == 0.0f)
        n = snprintf(out, (unsigned)maxLen, "0");
    else if (v < 0.0f)
        n = snprintf(out, (unsigned)maxLen, "%.4f", v);
    else if (v == (float)(int)v)
        n = snprintf(out, (unsigned)maxLen, "%d", (int)v);
    else
        n = snprintf(out, (unsigned)maxLen, "%.4f", v);
    if (n >= maxLen)
        out[maxLen - 1] = '\0';
}

/*---------------------------------------------------------------------------------
 * Fill rectangle in sub-screen bitmap (software rendering).
 *--------------------------------------------------------------------------------*/
static void drawRect(int x0, int y0, int x1, int y1, u16 color)
{
    int x, y;
    if (x0 < 0)
        x0 = 0;
    if (y0 < 0)
        y0 = 0;
    if (x1 > SUB_BMP_WIDTH)
        x1 = SUB_BMP_WIDTH;
    if (y1 > SUB_BMP_HEIGHT)
        y1 = SUB_BMP_HEIGHT;
    for (y = y0; y < y1; y++)
        for (x = x0; x < x1; x++)
            subBuffer[y * SUB_BMP_WIDTH + x] = color;
}

/* Sign of cross product for point-in-triangle test */
static int sign(int x0, int y0, int x1, int y1, int px, int py)
{
    return (x1 - x0) * (py - y0) - (y1 - y0) * (px - x0);
}

/* Fill triangle (software rendering). */
static void drawTriangle(int x0, int y0, int x1, int y1, int x2, int y2, u16 color)
{
    int minX = x0, maxX = x0, minY = y0, maxY = y0;
    int x, y;
    if (x1 < minX)
        minX = x1;
    if (x1 > maxX)
        maxX = x1;
    if (x2 < minX)
        minX = x2;
    if (x2 > maxX)
        maxX = x2;
    if (y1 < minY)
        minY = y1;
    if (y1 > maxY)
        maxY = y1;
    if (y2 < minY)
        minY = y2;
    if (y2 > maxY)
        maxY = y2;
    if (minX < 0)
        minX = 0;
    if (minY < 0)
        minY = 0;
    if (maxX >= SUB_BMP_WIDTH)
        maxX = SUB_BMP_WIDTH - 1;
    if (maxY >= SUB_BMP_HEIGHT)
        maxY = SUB_BMP_HEIGHT - 1;
    for (y = minY; y <= maxY; y++)
        for (x = minX; x <= maxX; x++)
        {
            int s0 = sign(x1, y1, x2, y2, x, y);
            int s1 = sign(x2, y2, x0, y0, x, y);
            int s2 = sign(x0, y0, x1, y1, x, y);
            /* Accept both winding orders so left and right arrows both fill */
            if ((s0 >= 0 && s1 >= 0 && s2 >= 0) || (s0 <= 0 && s1 <= 0 && s2 <= 0))
                subBuffer[y * SUB_BMP_WIDTH + x] = color;
        }
}

/* Return 1 if token matches (case-insensitive). */
static int tokenEquals(const char *line, const char *token);

/*---------------------------------------------------------------------------------
 * Script helpers: init from default, insert line, remove line.
 *--------------------------------------------------------------------------------*/
static int isRegToken(const char *token)
{
    return tokenEquals(token, "set") || tokenEquals(token, "add") || tokenEquals(token, "subtract") || tokenEquals(token, "multiply");
}

static int hasTwoNumberParams(const char *token)
{
    return tokenEquals(token, "if_gt") || tokenEquals(token, "if_lt") || tokenEquals(token, "cam_angle");
}

static void scriptLoadDefault(void)
{
    int i;
    scriptLines = defaultScriptLen;
    for (i = 0; i < scriptLines; i++)
    {
        strncpy(script[i], defaultScript[i], MAX_TOKEN_LEN - 1);
        script[i][MAX_TOKEN_LEN - 1] = '\0';
        if (tokenEquals(script[i], "sleep"))
            scriptParam[i] = 1.0f;
        else if (tokenEquals(script[i], "rotate"))
            scriptParam[i] = 0.0f, scriptParam2[i] = 1.0f; /* model index 0, amount */
        else if (tokenEquals(script[i], "model") || tokenEquals(script[i], "next_color") || tokenEquals(script[i], "background") || tokenEquals(script[i], "reset_model") || tokenEquals(script[i], "position") || tokenEquals(script[i], "angle"))
            scriptParam[i] = 0.0f; /* model/index */
        else if (tokenEquals(script[i], "cam_pos"))
            scriptParam[i] = 0.0f, scriptParam2[i] = 0.0f, scriptParam3[i] = 4.0f;
        else if (tokenEquals(script[i], "cam_angle"))
            scriptParam[i] = 0.0f, scriptParam2[i] = 0.0f;
        else if (tokenEquals(script[i], "if_true"))
            scriptParam[i] = 0.0f, scriptParamIsReg[i] = 1, scriptParamReg[i] = 0; /* default: register A; overrides set D/E/kA/Up/Dn for demo */
        else if (isRegToken(script[i]))
            scriptParam[i] = 0.0f, scriptReg[i] = 0;
        else
            scriptParam[i] = 0.0f;
        if (!isRegToken(script[i]) && !tokenEquals(script[i], "if_true"))
            scriptReg[i] = 0;
        if (!tokenEquals(script[i], "if_true"))
            scriptParamIsReg[i] = 0;
        scriptParamReg[i] = 0;
        scriptParam2[i] = 0.0f;
        scriptParamIsReg2[i] = 0;
        scriptParamReg2[i] = 0;
        scriptParam3[i] = 0.0f;
        scriptParamIsReg3[i] = 0;
        scriptParamReg3[i] = 0;
        scriptParam4[i] = 0.0f;
        scriptParamIsReg4[i] = 0;
        scriptParamReg4[i] = 0;
    }
    /* Overrides: IF_GT/IF_LT left = register, right = literal; IF_TRUE = register; SET/ADD/SUBTRACT reg + value; POSITION 4th param can be reg. */
    scriptParam[0] = 0.0f;
    scriptParam[1] = 1.0f;
    scriptParam[2] = 0.0f;
    scriptParam[3] = 0.0f;
    scriptParam[4] = 0.0f;
    scriptParam2[4] = 8.0f;
    scriptParam3[4] = 18.0f; /* CAM_POS 0 8 18 */
    scriptParam[5] = 0.0f;
    scriptParam2[5] = 25.0f; /* CAM_ANGLE 0 25 */
    scriptParam[6] = 2.0f;   /* BACKGROUND 2 */
    scriptReg[7] = 2;
    scriptParam[7] = 0.0f; /* SET C 0 */
    scriptParam[8] = 1.0f;
    scriptParam2[8] = -13.0f;
    scriptParam3[8] = 0.0f;
    scriptParam4[8] = 0.0f;
    scriptParamIsReg4[8] = 1;
    scriptParamReg4[8] = 2; /* POSITION 1 -13 0 C */
    scriptReg[9] = 0;
    scriptParam[9] = 0.0f; /* SET A 0 */
    scriptReg[10] = 1;
    scriptParam[10] = 1.0f; /* SET B 1 */
    scriptParamIsReg[12] = 1;
    scriptParamReg[12] = 5;
    scriptParam2[12] = 1.0f; /* IF_LT F 1 (only move ball when not game over) */
    scriptReg[13] = 0;
    scriptParamIsReg[13] = 1;
    scriptParamReg[13] = 1; /* ADD A B */
    scriptParamIsReg[15] = 1;
    scriptParamReg[15] = 0;
    scriptParam2[15] = 10.0f; /* IF_GT A 10 */
    scriptReg[16] = 0;
    scriptParam[16] = 10.0f; /* SET A 10 */
    scriptReg[17] = 1;
    scriptParam[17] = -1.0f; /* MULTIPLY B -1 */
    scriptParamIsReg[20] = 1;
    scriptParamReg[20] = 5;
    scriptParam2[20] = 1.0f; /* IF_LT F 1 (only run miss detection when not game over) */
    scriptReg[21] = 3;
    scriptParam[21] = 0.0f; /* SET D 0 */
    scriptParamIsReg[22] = 1;
    scriptParamReg[22] = 0;
    scriptParam2[22] = -10.0f; /* IF_LT A -10 */
    scriptParamIsReg[23] = 1;
    scriptParamReg[23] = 2;
    scriptParam2[23] = 2.0f; /* IF_GT C 2 */
    scriptReg[24] = 3;
    scriptParam[24] = 1.0f; /* SET D 1 */
    scriptParamIsReg[26] = 1;
    scriptParamReg[26] = 2;
    scriptParam2[26] = -2.0f; /* IF_LT C -2 */
    scriptReg[27] = 3;
    scriptParam[27] = 1.0f; /* SET D 1 */
    scriptParamIsReg[29] = 1;
    scriptParamReg[29] = 3; /* IF_TRUE D (miss: start game over) */
    scriptReg[30] = 5;
    scriptParam[30] = 60.0f; /* SET F 60 (game-over countdown) */
    scriptReg[31] = 6;
    scriptParam[31] = 3.0f; /* SET G 3 (beeps left) */
    scriptReg[33] = 4;
    scriptParam[33] = 1.0f; /* SET E 1 */
    scriptReg[34] = 4;
    scriptParamIsReg[34] = 1;
    scriptParamReg[34] = 3; /* SUBTRACT E D */
    scriptParamIsReg[35] = 1;
    scriptParamReg[35] = 4; /* IF_TRUE E (hit: bounce) */
    scriptReg[36] = 0;
    scriptParam[36] = -10.0f; /* SET A -10 */
    scriptReg[37] = 1;
    scriptParam[37] = -1.0f; /* MULTIPLY B -1 */
    scriptParamIsReg[42] = 1;
    scriptParamReg[42] = 5;
    scriptParam2[42] = 0.0f; /* IF_GT F 0 (game-over active) */
    scriptReg[43] = 5;
    scriptParam[43] = 1.0f; /* SUBTRACT F 1 */
    scriptParamIsReg[44] = 1;
    scriptParamReg[44] = 6; /* IF_TRUE G (beep this frame) */
    scriptReg[46] = 6;
    scriptParam[46] = 1.0f; /* SUBTRACT G 1 */
    scriptParamIsReg[49] = 1;
    scriptParamReg[49] = 5;
    scriptParam2[49] = 1.0f; /* IF_LT F 1 (F==0: do reset check) line 49 */
    scriptParamIsReg[50] = 1;
    scriptParamReg[50] = 3; /* IF_TRUE D (reset ball) line 50 */
    scriptReg[51] = 0;
    scriptParam[51] = 0.0f; /* SET A 0 line 51 */
    scriptReg[52] = 1;
    scriptParam[52] = 1.0f; /* SET B 1 line 52 */
    scriptReg[53] = 3;
    scriptParam[53] = 0.0f; /* SET D 0 line 53 */
    scriptParamIsReg[56] = 1;
    scriptParamReg[56] = 5;
    scriptParam2[56] = 1.0f; /* IF_LT F 1 (only paddle input when not game over) line 56 */
    scriptParamIsReg[57] = 1;
    scriptParamReg[57] = REG_BTN_A; /* IF_TRUE kA line 57 */
    scriptParam[58] = 0.0f;         /* NEXT_COLOR 0 line 58 */
    scriptParamIsReg[60] = 1;
    scriptParamReg[60] = REG_UP; /* IF_TRUE Up line 60 */
    scriptReg[61] = 2;
    scriptParam[61] = -0.5f; /* ADD C -0.5 line 61 */
    scriptParamIsReg[63] = 1;
    scriptParamReg[63] = REG_DN; /* IF_TRUE Dn line 63 */
    scriptReg[64] = 2;
    scriptParam[64] = 0.5f; /* ADD C 0.5 line 64 */
    scriptParam[67] = 0.0f;
    scriptParamIsReg2[67] = 1;
    scriptParamReg2[67] = 0;
    scriptParam3[67] = 0.0f;
    scriptParam4[67] = 0.0f; /* POSITION 0 A 0 0 (ball) */
    scriptParam[68] = 1.0f;
    scriptParam2[68] = -13.0f;
    scriptParam3[68] = 0.0f;
    scriptParam4[68] = 0.0f;
    scriptParamIsReg4[68] = 1;
    scriptParamReg4[68] = 2;  /* POSITION 1 -13 0 C (paddle) */
    scriptParam[69] = 0.016f; /* SLEEP 0.016 */
    for (i = scriptLines; i < MAX_SCRIPT_LINES; i++)
    {
        script[i][0] = '\0';
        scriptParam[i] = 0.0f;
        scriptReg[i] = 0;
        scriptParamIsReg[i] = 0;
        scriptParamReg[i] = 0;
        scriptParam2[i] = 0.0f;
        scriptParamIsReg2[i] = 0;
        scriptParamReg2[i] = 0;
        scriptParam3[i] = 0.0f;
        scriptParamIsReg3[i] = 0;
        scriptParamReg3[i] = 0;
        scriptParam4[i] = 0.0f;
        scriptParamIsReg4[i] = 0;
        scriptParamReg4[i] = 0;
    }
    selectedLine = 0;
}

/* Copy current editor to slot (used before switching or on exit). */
static void scriptSaveToSlot(int slot)
{
    int i;
    if (slot < 0 || slot >= NUM_SCRIPT_SLOTS)
        return;
    scriptSlotLines[slot] = scriptLines;
    for (i = 0; i < scriptLines; i++)
    {
        strncpy(scriptSlot[slot][i], script[i], MAX_TOKEN_LEN - 1);
        scriptSlot[slot][i][MAX_TOKEN_LEN - 1] = '\0';
        scriptParamSlot[slot][i] = scriptParam[i];
        scriptRegSlot[slot][i] = scriptReg[i];
        scriptParamIsRegSlot[slot][i] = scriptParamIsReg[i];
        scriptParamRegSlot[slot][i] = scriptParamReg[i];
        scriptParam2Slot[slot][i] = scriptParam2[i];
        scriptParamIsReg2Slot[slot][i] = scriptParamIsReg2[i];
        scriptParamReg2Slot[slot][i] = scriptParamReg2[i];
        scriptParam3Slot[slot][i] = scriptParam3[i];
        scriptParamIsReg3Slot[slot][i] = scriptParamIsReg3[i];
        scriptParamReg3Slot[slot][i] = scriptParamReg3[i];
        scriptParam4Slot[slot][i] = scriptParam4[i];
        scriptParamIsReg4Slot[slot][i] = scriptParamIsReg4[i];
        scriptParamReg4Slot[slot][i] = scriptParamReg4[i];
    }
}

/* Copy slot into editor (used when picking a script from menu). */
static void scriptLoadFromSlot(int slot)
{
    int i;
    if (slot < 0 || slot >= NUM_SCRIPT_SLOTS)
        return;
    scriptLines = scriptSlotLines[slot];
    for (i = 0; i < scriptLines; i++)
    {
        strncpy(script[i], scriptSlot[slot][i], MAX_TOKEN_LEN - 1);
        script[i][MAX_TOKEN_LEN - 1] = '\0';
        scriptParam[i] = scriptParamSlot[slot][i];
        scriptReg[i] = scriptRegSlot[slot][i];
        scriptParamIsReg[i] = scriptParamIsRegSlot[slot][i];
        scriptParamReg[i] = scriptParamRegSlot[slot][i];
        scriptParam2[i] = scriptParam2Slot[slot][i];
        scriptParamIsReg2[i] = scriptParamIsReg2Slot[slot][i];
        scriptParamReg2[i] = scriptParamReg2Slot[slot][i];
        scriptParam3[i] = scriptParam3Slot[slot][i];
        scriptParamIsReg3[i] = scriptParamIsReg3Slot[slot][i];
        scriptParamReg3[i] = scriptParamReg3Slot[slot][i];
        scriptParam4[i] = scriptParam4Slot[slot][i];
        scriptParamIsReg4[i] = scriptParamIsReg4Slot[slot][i];
        scriptParamReg4[i] = scriptParamReg4Slot[slot][i];
    }
    for (i = scriptLines; i < MAX_SCRIPT_LINES; i++)
        script[i][0] = '\0';
    selectedLine = 0;
    scriptScrollOffset = 0;
}

static void scriptResetExecution(void)
{
    int i;
    scriptRunning = 0;
    scriptIP = 0;
    loopSp = 0;
    sleepFramesLeft = 0;
    for (i = 0; i < NUM_REGISTERS; i++)
        registers[i] = 0.0f;
    for (i = 0; i < MAX_MODELS; i++)
    {
        modelActive[i] = 0;
        modelAngle[i] = 0.0f;
        modelX[i] = modelY[i] = modelZ[i] = 0.0f;
        modelColorIndex[i] = 0;
    }
}

/* Insert token as new line after selectedLine (each token on its own line). */
static void scriptInsertLine(const char *token)
{
    int i;
    if (scriptLines >= MAX_SCRIPT_LINES)
        return;
    if (scriptLines == 0)
    {
        strncpy(script[0], token, MAX_TOKEN_LEN - 1);
        script[0][MAX_TOKEN_LEN - 1] = '\0';
        scriptParam[0] = tokenEquals(token, "sleep") ? 1.0f : (tokenEquals(token, "rotate") ? 0.0f : (tokenEquals(token, "model") || tokenEquals(token, "next_color") || tokenEquals(token, "background") || tokenEquals(token, "reset_model") || tokenEquals(token, "position") || tokenEquals(token, "angle") ? 0.0f : 0.0f)); /* model/index */
        scriptReg[0] = isRegToken(token) ? 0 : 0;
        scriptParamIsReg[0] = 0;
        scriptParamReg[0] = 0;
        scriptParam2[0] = tokenEquals(token, "rotate") ? 1.0f : 0.0f; /* ROTATE amount */
        scriptParamIsReg2[0] = 0;
        scriptParamReg2[0] = 0;
        scriptParam3[0] = tokenEquals(token, "cam_pos") ? 4.0f : 0.0f; /* CAM_POS z default 4 */
        scriptParamIsReg3[0] = 0;
        scriptParamReg3[0] = 0;
        scriptParam4[0] = 0.0f;
        scriptParamIsReg4[0] = 0;
        scriptParamReg4[0] = 0;
        scriptLines = 1;
        selectedLine = 0; /* cursor on the new token */
        return;
    }
    for (i = scriptLines; i > selectedLine + 1; i--)
    {
        strncpy(script[i], script[i - 1], MAX_TOKEN_LEN - 1);
        script[i][MAX_TOKEN_LEN - 1] = '\0';
        scriptParam[i] = scriptParam[i - 1];
        scriptReg[i] = scriptReg[i - 1];
        scriptParamIsReg[i] = scriptParamIsReg[i - 1];
        scriptParamReg[i] = scriptParamReg[i - 1];
        scriptParam2[i] = scriptParam2[i - 1];
        scriptParamIsReg2[i] = scriptParamIsReg2[i - 1];
        scriptParamReg2[i] = scriptParamReg2[i - 1];
        scriptParam3[i] = scriptParam3[i - 1];
        scriptParamIsReg3[i] = scriptParamIsReg3[i - 1];
        scriptParamReg3[i] = scriptParamReg3[i - 1];
        scriptParam4[i] = scriptParam4[i - 1];
        scriptParamIsReg4[i] = scriptParamIsReg4[i - 1];
        scriptParamReg4[i] = scriptParamReg4[i - 1];
    }
    strncpy(script[selectedLine + 1], token, MAX_TOKEN_LEN - 1);
    script[selectedLine + 1][MAX_TOKEN_LEN - 1] = '\0';
    scriptParam[selectedLine + 1] = tokenEquals(token, "sleep") ? 1.0f : (tokenEquals(token, "rotate") ? 0.0f : (tokenEquals(token, "model") || tokenEquals(token, "next_color") || tokenEquals(token, "background") || tokenEquals(token, "reset_model") || tokenEquals(token, "position") || tokenEquals(token, "angle") ? 0.0f : 0.0f));
    scriptReg[selectedLine + 1] = isRegToken(token) ? 0 : 0;
    scriptParamIsReg[selectedLine + 1] = 0;
    scriptParamReg[selectedLine + 1] = 0;
    scriptParam2[selectedLine + 1] = tokenEquals(token, "rotate") ? 1.0f : 0.0f;
    scriptParamIsReg2[selectedLine + 1] = 0;
    scriptParamReg2[selectedLine + 1] = 0;
    scriptParam3[selectedLine + 1] = tokenEquals(token, "cam_pos") ? 4.0f : 0.0f;
    scriptParamIsReg3[selectedLine + 1] = 0;
    scriptParamReg3[selectedLine + 1] = 0;
    scriptParam4[selectedLine + 1] = 0.0f;
    scriptParamIsReg4[selectedLine + 1] = 0;
    scriptParamReg4[selectedLine + 1] = 0;
    scriptLines++;
    selectedLine = selectedLine + 1; /* cursor on the new token's index */
}

/* Clear current line and collapse. */
static void scriptClearLine(void)
{
    int i;
    if (scriptLines <= 0)
        return;
    if (selectedLine < 0)
        selectedLine = 0; /* index -1 = insertion above first; clear first line (0) */
    for (i = selectedLine; i < scriptLines - 1; i++)
    {
        strncpy(script[i], script[i + 1], MAX_TOKEN_LEN - 1);
        script[i][MAX_TOKEN_LEN - 1] = '\0';
        scriptParam[i] = scriptParam[i + 1];
        scriptReg[i] = scriptReg[i + 1];
        scriptParamIsReg[i] = scriptParamIsReg[i + 1];
        scriptParamReg[i] = scriptParamReg[i + 1];
        scriptParam2[i] = scriptParam2[i + 1];
        scriptParamIsReg2[i] = scriptParamIsReg2[i + 1];
        scriptParamReg2[i] = scriptParamReg2[i + 1];
        scriptParam3[i] = scriptParam3[i + 1];
        scriptParamIsReg3[i] = scriptParamIsReg3[i + 1];
        scriptParamReg3[i] = scriptParamReg3[i + 1];
        scriptParam4[i] = scriptParam4[i + 1];
        scriptParamIsReg4[i] = scriptParamIsReg4[i + 1];
        scriptParamReg4[i] = scriptParamReg4[i + 1];
    }
    script[scriptLines - 1][0] = '\0';
    scriptParam[scriptLines - 1] = 0.0f;
    scriptReg[scriptLines - 1] = 0;
    scriptParamIsReg[scriptLines - 1] = 0;
    scriptParamReg[scriptLines - 1] = 0;
    scriptParam2[scriptLines - 1] = 0.0f;
    scriptParamIsReg2[scriptLines - 1] = 0;
    scriptParamReg2[scriptLines - 1] = 0;
    scriptParam3[scriptLines - 1] = 0.0f;
    scriptParamIsReg3[scriptLines - 1] = 0;
    scriptParamReg3[scriptLines - 1] = 0;
    scriptParam4[scriptLines - 1] = 0.0f;
    scriptParamIsReg4[scriptLines - 1] = 0;
    scriptParamReg4[scriptLines - 1] = 0;
    scriptLines--;
    if (selectedLine >= scriptLines && scriptLines > 0)
        selectedLine = scriptLines - 1;
}

static int tokenEquals(const char *line, const char *token)
{
    while (*line && *token)
    {
        char a = *line, b = *token;
        if (a >= 'a' && a <= 'z')
            a -= 32;
        if (b >= 'a' && b <= 'z')
            b -= 32;
        if (a != b)
            return 0;
        line++;
        token++;
    }
    return *line == *token;
}

/*---------------------------------------------------------------------------------
 * UI layout: top row [play/pause][stop][step], script area (fills), nav [up][down][token][clear].
 * Tokens are in a separate view (tokenViewActive); tap Token to open.
 *--------------------------------------------------------------------------------*/
#define TOP_Y0 2
#define TOP_Y1 22
#define BTN_W 82
#define STOP_ICON_HALF 6 /* stop square 12x12 so button styling shows */
#define STOP_BTN_X0 (2 + BTN_W + 2)
#define STOP_BTN_X1 (2 + 2 * (BTN_W) + 2)
#define STEP_BTN_X0 (2 + 2 * (BTN_W + 2))
#define STEP_BTN_W 54
#define STEP_BTN_X1 (STEP_BTN_X0 + STEP_BTN_W)
#define SCRIPTS_BTN_X0 (STEP_BTN_X1 + 2)
#define SCRIPTS_BTN_X1 (SUB_BMP_WIDTH - 2)
#define SCRIPT_Y0 26
#define LINE_H 10
#define SCRIPT_LINES_VIS 14
#define SCRIPT_Y1 (SCRIPT_Y0 + SCRIPT_LINES_VIS * LINE_H)
#define NAV_Y0 (SCRIPT_Y1 + 2)
#define NAV_Y1 (NAV_Y0 + 20)
#define NAV_BTN_W 61
/* Token view overlay layout (when tokenViewActive) */
#define TV_BACK_X0 2
#define TV_BACK_Y0 2
#define TV_BACK_X1 26
#define TV_BACK_Y1 22
#define TV_ROW0_Y 28
#define TV_ROW_H 19
#define TV_ROW1_Y (TV_ROW0_Y + TV_ROW_H + 2)
#define TV_ROW2_Y (TV_ROW1_Y + TV_ROW_H + 2)
#define TV_ROW3_Y (TV_ROW2_Y + TV_ROW_H + 2)
#define TV_ROW4_Y (TV_ROW3_Y + TV_ROW_H + 2)
#define TV_ROW5_Y (TV_ROW4_Y + TV_ROW_H + 2)
#define TV_BTN_W 61
#define TV_TOKENS_PER_ROW 4

/* Script menu overlay: back + Script 1..6 */
#define SM_BACK_X0 2
#define SM_BACK_Y0 2
#define SM_BACK_X1 26
#define SM_BACK_Y1 22
#define SM_ROW0_Y 28
#define SM_ROW_H 24
#define SM_BTN_W (SUB_BMP_WIDTH - 4)

static void drawButton(int x0, int y0, int x1, int y1)
{
    drawRect(x0, y0, x1, y1, COLOR_BTN);
    drawRect(x0, y0, x1, y0 + 2, COLOR_BTN_EDGE);
    drawRect(x0, y1 - 2, x1, y1, COLOR_BTN_EDGE);
    drawRect(x0, y0, x0 + 2, y1, COLOR_BTN_EDGE);
    drawRect(x1 - 2, y0, x1, y1, COLOR_BTN_EDGE);
}

/* Play icon: triangle pointing right (slightly smaller). Pause: two vertical bars. */
static void drawPlayIcon(int cx, int cy)
{
    drawTriangle(cx + 5, cy, cx - 5, cy - 6, cx - 5, cy + 6, COLOR_ARROW);
}

static void drawPauseIcon(int cx, int cy)
{
    drawRect(cx - 5, cy - 6, cx - 2, cy + 6, COLOR_ARROW);
    drawRect(cx + 2, cy - 6, cx + 5, cy + 6, COLOR_ARROW);
}

static void drawStopIcon(int cx, int cy)
{
    int h = STOP_ICON_HALF;
    drawRect(cx - h, cy - h, cx + h, cy + h, COLOR_ARROW);
}

/* Up/down arrows (~20% smaller triangles). */
static void drawUpArrow(int cx, int cy)
{
    drawTriangle(cx, cy - 4, cx - 5, cy + 4, cx + 5, cy + 4, COLOR_ARROW);
}

static void drawDownArrow(int cx, int cy)
{
    drawTriangle(cx, cy + 4, cx - 5, cy - 4, cx + 5, cy - 4, COLOR_ARROW);
}

/* Redraw entire bottom screen: scripting editor. */
static void drawSubScreen(void)
{
    int i, x, y;

    dmaFillHalfWords(COLOR_BG, subBuffer, SUB_BMP_WIDTH * SUB_VIS_HEIGHT * sizeof(u16));

    /* Top row: [play/pause] [stop] [step] */
    drawButton(2, TOP_Y0, 2 + BTN_W, TOP_Y1);
    if (scriptRunning)
        drawPauseIcon(2 + BTN_W / 2, (TOP_Y0 + TOP_Y1) / 2);
    else
        drawPlayIcon(2 + BTN_W / 2, (TOP_Y0 + TOP_Y1) / 2);

    drawButton(STOP_BTN_X0, TOP_Y0, STOP_BTN_X1, TOP_Y1);
    if (scriptActive)
        drawRect(STOP_BTN_X0, TOP_Y0, STOP_BTN_X1, TOP_Y1, COLOR_PLAYING);
    drawStopIcon((STOP_BTN_X0 + STOP_BTN_X1) / 2, (TOP_Y0 + TOP_Y1) / 2);

    drawButton(STEP_BTN_X0, TOP_Y0, STEP_BTN_X1, TOP_Y1);
    drawStringCentered(STEP_BTN_X0, TOP_Y0, STEP_BTN_X1, TOP_Y1, "STEP", COLOR_TEXT);

    drawButton(SCRIPTS_BTN_X0, TOP_Y0, SCRIPTS_BTN_X1, TOP_Y1);
    drawStringCentered(SCRIPTS_BTN_X0, TOP_Y0, SCRIPTS_BTN_X1, TOP_Y1, "1-6", COLOR_TEXT);

    /* Script area: when paused, only scroll to keep selection visible (don't recenter on insert). Allow scrollOffset -1 for blank row above first line. */
    if (!scriptRunning)
    {
        if (selectedLine < scriptScrollOffset)
            scriptScrollOffset = selectedLine;
        if (selectedLine >= scriptScrollOffset + SCRIPT_LINES_VIS)
            scriptScrollOffset = selectedLine - SCRIPT_LINES_VIS + 1;
        if (scriptScrollOffset + SCRIPT_LINES_VIS > scriptLines)
            scriptScrollOffset = scriptLines - SCRIPT_LINES_VIS;
        if (scriptScrollOffset < -1)
            scriptScrollOffset = -1; /* at most one blank row above (index -1) */
        if (scriptScrollOffset < 0 && selectedLine >= 0)
            scriptScrollOffset = 0; /* only clamp to 0 when not at index -1 */
    }
    for (i = 0; i < SCRIPT_LINES_VIS; i++)
    {
        int lineIdx = scriptScrollOffset + i;
        y = SCRIPT_Y0 + i * LINE_H;
        if (lineIdx == selectedLine)
            drawRect(2, y, SUB_BMP_WIDTH - 2, y + LINE_H, scriptRunning ? COLOR_HIGHLIGHT : COLOR_HIGHLIGHT_PAUSED);
        if (lineIdx >= 0 && lineIdx < scriptLines && script[lineIdx][0])
        {
            drawString(6, y + 1, script[lineIdx], COLOR_TEXT);
            if (tokenEquals(script[lineIdx], "model"))
            {
                int x0 = 6 + stringWidth("MODEL ") + (FONT_W + 1);
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    char buf[12];
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (tokenEquals(script[lineIdx], "sleep"))
            {
                int x0 = 6 + stringWidth("SLEEP ") + (FONT_W + 1);
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    char buf[12];
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (tokenEquals(script[lineIdx], "next_color"))
            {
                int x0 = 6 + stringWidth("NEXT_COLOR ") + (FONT_W + 1);
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    char buf[12];
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (tokenEquals(script[lineIdx], "rotate"))
            {
                int slotW = (FONT_W + 1) * 6;
                int x0 = 6 + stringWidth("ROTATE ") + (FONT_W + 1);
                char buf[12];
                /* model index */
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += slotW;
                /* amount */
                if (scriptParamIsReg2[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg2[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam2[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (isRegToken(script[lineIdx]))
            {
                char regLetter[2] = {'A' + (char)scriptReg[lineIdx], '\0'};
                int preW;
                if (tokenEquals(script[lineIdx], "set"))
                    preW = stringWidth("SET ");
                else if (tokenEquals(script[lineIdx], "add"))
                    preW = stringWidth("ADD ");
                else if (tokenEquals(script[lineIdx], "multiply"))
                    preW = stringWidth("MULTIPLY ");
                else
                    preW = stringWidth("SUBTRACT ");
                int x0 = 6 + preW;
                drawString(x0, y + 1, regLetter, COLOR_TEXT);
                x0 += (FONT_W + 1) * 2;
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    char buf[12];
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (tokenEquals(script[lineIdx], "beep"))
            {
                /* no params */
            }
            else if (tokenEquals(script[lineIdx], "cam_pos"))
            {
                int slotW = (FONT_W + 1) * 6;
                int x0 = 6 + stringWidth("CAM_POS ") + (FONT_W + 1);
                char buf[12];
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += slotW;
                if (scriptParamIsReg2[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg2[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam2[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += slotW;
                if (scriptParamIsReg3[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg3[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam3[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (tokenEquals(script[lineIdx], "cam_angle"))
            {
                int slotW = (FONT_W + 1) * 6;
                int x0 = 6 + stringWidth("CAM_ANGLE ") + (FONT_W + 1);
                char buf[12];
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += slotW;
                if (scriptParamIsReg2[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg2[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam2[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (tokenEquals(script[lineIdx], "background"))
            {
                int x0 = 6 + stringWidth("BACKGROUND ") + (FONT_W + 1);
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    char buf[12];
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (tokenEquals(script[lineIdx], "reset_model"))
            {
                int x0 = 6 + stringWidth("RESET_MODEL ") + (FONT_W + 1);
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    char buf[12];
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (tokenEquals(script[lineIdx], "translate"))
            {
                int slotW = (FONT_W + 1) * 6;
                int x0 = 6 + stringWidth("TRANSLATE ") + (FONT_W + 1);
                char buf[12];
                /* model index */
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += slotW;
                /* X */
                if (scriptParamIsReg2[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg2[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam2[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += slotW;
                /* Y */
                if (scriptParamIsReg3[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg3[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam3[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += slotW;
                /* Z */
                if (scriptParamIsReg4[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg4[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam4[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (tokenEquals(script[lineIdx], "position"))
            {
                int slotW = (FONT_W + 1) * 6;
                int x0 = 6 + stringWidth("POSITION ") + (FONT_W + 1);
                char buf[12];
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += slotW;
                if (scriptParamIsReg2[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg2[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam2[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += slotW;
                if (scriptParamIsReg3[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg3[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam3[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += slotW;
                if (scriptParamIsReg4[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg4[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam4[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (tokenEquals(script[lineIdx], "angle"))
            {
                int slotW = (FONT_W + 1) * 6;
                int x0 = 6 + stringWidth("ANGLE ") + (FONT_W + 1);
                char buf[12];
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += slotW;
                if (scriptParamIsReg2[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg2[lineIdx], COLOR_TEXT);
                else
                {
                    formatSleepParam(scriptParam2[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (tokenEquals(script[lineIdx], "if_true"))
            {
                int x0 = 6 + stringWidth("IF_TRUE ") + (FONT_W + 1);
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    char buf[12];
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
            else if (hasTwoNumberParams(script[lineIdx]))
            {
                int preW = tokenEquals(script[lineIdx], "if_gt") ? stringWidth("IF_GT ") : stringWidth("IF_LT ");
                int x0 = 6 + preW;
                if (scriptParamIsReg[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg[lineIdx], COLOR_TEXT);
                else
                {
                    char buf[12];
                    formatSleepParam(scriptParam[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
                x0 += (FONT_W + 1) * 4;
                if (scriptParamIsReg2[lineIdx])
                    drawRegisterAt(x0, y + 1, scriptParamReg2[lineIdx], COLOR_TEXT);
                else
                {
                    char buf[12];
                    formatSleepParam(scriptParam2[lineIdx], buf, sizeof(buf));
                    drawString(x0, y + 1, buf, COLOR_TEXT);
                }
            }
        }
    }

    /* Nav: [up] [down] [token] [clear] */
    x = 2;
    drawButton(x, NAV_Y0, x + NAV_BTN_W, NAV_Y1);
    drawUpArrow(x + NAV_BTN_W / 2, (NAV_Y0 + NAV_Y1) / 2);
    x += NAV_BTN_W + 2;
    drawButton(x, NAV_Y0, x + NAV_BTN_W, NAV_Y1);
    drawDownArrow(x + NAV_BTN_W / 2, (NAV_Y0 + NAV_Y1) / 2);
    x += NAV_BTN_W + 2;
    drawButton(x, NAV_Y0, x + NAV_BTN_W, NAV_Y1);
    drawStringCentered(x, NAV_Y0, x + NAV_BTN_W, NAV_Y1, "TOKEN", COLOR_TEXT);
    x += NAV_BTN_W + 2;
    drawButton(x, NAV_Y0, x + NAV_BTN_W, NAV_Y1);
    drawStringCentered(x, NAV_Y0, x + NAV_BTN_W, NAV_Y1, "CLEAR", COLOR_TEXT);
}

/* Token view: Back button + all tokens in 3 rows. Hit: 0=none, 1=back, 7-19=token. */
static void drawTokenViewScreen(void)
{
    int x;
    dmaFillHalfWords(COLOR_BG, subBuffer, SUB_BMP_WIDTH * SUB_VIS_HEIGHT * sizeof(u16));
    drawButton(TV_BACK_X0, TV_BACK_Y0, TV_BACK_X1, TV_BACK_Y1);
    {
        int cx = (TV_BACK_X0 + TV_BACK_X1) / 2, cy = (TV_BACK_Y0 + TV_BACK_Y1) / 2;
        drawTriangle(cx - 6, cy, cx + 4, cy - 6, cx + 4, cy + 6, COLOR_ARROW);
    }
    /* 4 tokens per row. Row 0: NEXT_COLOR, SLEEP, LOOP, END_LOOP */
    x = 2;
    drawButton(x, TV_ROW0_Y, x + TV_BTN_W, TV_ROW0_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW0_Y, x + TV_BTN_W, TV_ROW0_Y + TV_ROW_H, "NEXT_COLOR", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW0_Y, x + TV_BTN_W, TV_ROW0_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW0_Y, x + TV_BTN_W, TV_ROW0_Y + TV_ROW_H, "SLEEP", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW0_Y, x + TV_BTN_W, TV_ROW0_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW0_Y, x + TV_BTN_W, TV_ROW0_Y + TV_ROW_H, "LOOP", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW0_Y, x + TV_BTN_W, TV_ROW0_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW0_Y, x + TV_BTN_W, TV_ROW0_Y + TV_ROW_H, "END_LOOP", COLOR_TEXT);
    /* Row 1: SET, ADD, SUBTRACT, ROTATE */
    x = 2;
    drawButton(x, TV_ROW1_Y, x + TV_BTN_W, TV_ROW1_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW1_Y, x + TV_BTN_W, TV_ROW1_Y + TV_ROW_H, "SET", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW1_Y, x + TV_BTN_W, TV_ROW1_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW1_Y, x + TV_BTN_W, TV_ROW1_Y + TV_ROW_H, "ADD", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW1_Y, x + TV_BTN_W, TV_ROW1_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW1_Y, x + TV_BTN_W, TV_ROW1_Y + TV_ROW_H, "SUBTRACT", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW1_Y, x + TV_BTN_W, TV_ROW1_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW1_Y, x + TV_BTN_W, TV_ROW1_Y + TV_ROW_H, "ROTATE", COLOR_TEXT);
    /* Row 2: MULTIPLY, TRANSLATE, IF_GT, IF_LT */
    x = 2;
    drawButton(x, TV_ROW2_Y, x + TV_BTN_W, TV_ROW2_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW2_Y, x + TV_BTN_W, TV_ROW2_Y + TV_ROW_H, "MULTIPLY", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW2_Y, x + TV_BTN_W, TV_ROW2_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW2_Y, x + TV_BTN_W, TV_ROW2_Y + TV_ROW_H, "TRANSLATE", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW2_Y, x + TV_BTN_W, TV_ROW2_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW2_Y, x + TV_BTN_W, TV_ROW2_Y + TV_ROW_H, "IF_GT", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW2_Y, x + TV_BTN_W, TV_ROW2_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW2_Y, x + TV_BTN_W, TV_ROW2_Y + TV_ROW_H, "IF_LT", COLOR_TEXT);
    /* Row 3: MODEL, IF_TRUE, END_IF */
    x = 2;
    drawButton(x, TV_ROW3_Y, x + TV_BTN_W, TV_ROW3_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW3_Y, x + TV_BTN_W, TV_ROW3_Y + TV_ROW_H, "MODEL", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW3_Y, x + TV_BTN_W, TV_ROW3_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW3_Y, x + TV_BTN_W, TV_ROW3_Y + TV_ROW_H, "IF_TRUE", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW3_Y, x + TV_BTN_W, TV_ROW3_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW3_Y, x + TV_BTN_W, TV_ROW3_Y + TV_ROW_H, "END_IF", COLOR_TEXT);
    /* Row 4: BEEP, CAM_POS, CAM_ANGLE */
    x = 2;
    drawButton(x, TV_ROW4_Y, x + TV_BTN_W, TV_ROW4_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW4_Y, x + TV_BTN_W, TV_ROW4_Y + TV_ROW_H, "BEEP", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW4_Y, x + TV_BTN_W, TV_ROW4_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW4_Y, x + TV_BTN_W, TV_ROW4_Y + TV_ROW_H, "CAM_POS", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW4_Y, x + TV_BTN_W, TV_ROW4_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW4_Y, x + TV_BTN_W, TV_ROW4_Y + TV_ROW_H, "CAM_ANGLE", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW4_Y, x + TV_BTN_W, TV_ROW4_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW4_Y, x + TV_BTN_W, TV_ROW4_Y + TV_ROW_H, "BACKGROUND", COLOR_TEXT);
    /* Row 5: RESET_MODEL, POSITION, ANGLE */
    x = 2;
    drawButton(x, TV_ROW5_Y, x + TV_BTN_W, TV_ROW5_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW5_Y, x + TV_BTN_W, TV_ROW5_Y + TV_ROW_H, "RESET_MODEL", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW5_Y, x + TV_BTN_W, TV_ROW5_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW5_Y, x + TV_BTN_W, TV_ROW5_Y + TV_ROW_H, "POSITION", COLOR_TEXT);
    x += TV_BTN_W + 2;
    drawButton(x, TV_ROW5_Y, x + TV_BTN_W, TV_ROW5_Y + TV_ROW_H);
    drawStringCentered(x, TV_ROW5_Y, x + TV_BTN_W, TV_ROW5_Y + TV_ROW_H, "ANGLE", COLOR_TEXT);
}

/* Script menu: back + Script 1..6. Hit: 0=none, 1=back, 2..7=script 1..6. */
static void drawScriptMenuScreen(void)
{
    int i, y;
    char buf[24];
    dmaFillHalfWords(COLOR_BG, subBuffer, SUB_BMP_WIDTH * SUB_VIS_HEIGHT * sizeof(u16));
    drawButton(SM_BACK_X0, SM_BACK_Y0, SM_BACK_X1, SM_BACK_Y1);
    {
        int cx = (SM_BACK_X0 + SM_BACK_X1) / 2, cy = (SM_BACK_Y0 + SM_BACK_Y1) / 2;
        drawTriangle(cx - 6, cy, cx + 4, cy - 6, cx + 4, cy + 6, COLOR_ARROW);
    }
    for (i = 0; i < NUM_SCRIPT_SLOTS; i++)
    {
        y = SM_ROW0_Y + i * (SM_ROW_H + 2);
        drawButton(2, y, 2 + SM_BTN_W, y + SM_ROW_H);
        if (i == currentScriptSlot)
            drawRect(2, y, 2 + SM_BTN_W, y + SM_ROW_H, COLOR_HIGHLIGHT_PAUSED);
        sprintf(buf, "Script %d", i + 1);
        if (scriptSlotLines[i] > 0)
        {
            char ln[12];
            sprintf(ln, " (%d)", scriptSlotLines[i]);
            strcat(buf, ln);
        }
        drawStringCentered(2, y, 2 + SM_BTN_W, y + SM_ROW_H, buf, COLOR_TEXT);
    }
}

static int scriptMenuHitTest(int px, int py)
{
    int i, y;
    if (py >= SM_BACK_Y0 && py < SM_BACK_Y1 && px >= SM_BACK_X0 && px < SM_BACK_X1)
        return 1;
    for (i = 0; i < NUM_SCRIPT_SLOTS; i++)
    {
        y = SM_ROW0_Y + i * (SM_ROW_H + 2);
        if (py >= y && py < y + SM_ROW_H && px >= 2 && px < 2 + SM_BTN_W)
            return 2 + i; /* 2..7 = script 1..6 */
    }
    return 0;
}

/* Token view hit: 0=none, 1=back, 7-19=token. 4 tokens per row. */
static int tokenViewHitTest(int px, int py)
{
    int x, col;
    if (py >= TV_BACK_Y0 && py < TV_BACK_Y1 && px >= TV_BACK_X0 && px < TV_BACK_X1)
        return 1;
    if (py >= TV_ROW0_Y && py < TV_ROW0_Y + TV_ROW_H)
    {
        x = 2;
        for (col = 0; col < 4; col++)
        {
            if (px >= x && px < x + TV_BTN_W)
                return 7 + col;
            x += TV_BTN_W + 2;
        }
    }
    if (py >= TV_ROW1_Y && py < TV_ROW1_Y + TV_ROW_H)
    {
        x = 2;
        for (col = 0; col < 4; col++)
        {
            if (px >= x && px < x + TV_BTN_W)
                return 11 + col;
            x += TV_BTN_W + 2;
        }
    }
    if (py >= TV_ROW2_Y && py < TV_ROW2_Y + TV_ROW_H)
    {
        x = 2;
        for (col = 0; col < 4; col++)
        {
            if (px >= x && px < x + TV_BTN_W)
                return 15 + col;
            x += TV_BTN_W + 2;
        }
    }
    if (py >= TV_ROW3_Y && py < TV_ROW3_Y + TV_ROW_H)
    {
        x = 2;
        if (px >= x && px < x + TV_BTN_W)
            return 19; /* MODEL */
        x += TV_BTN_W + 2;
        if (px >= x && px < x + TV_BTN_W)
            return 20; /* IF_TRUE */
        x += TV_BTN_W + 2;
        if (px >= x && px < x + TV_BTN_W)
            return 21; /* END_IF */
    }
    if (py >= TV_ROW4_Y && py < TV_ROW4_Y + TV_ROW_H)
    {
        x = 2;
        if (px >= x && px < x + TV_BTN_W)
            return 22; /* BEEP */
        x += TV_BTN_W + 2;
        if (px >= x && px < x + TV_BTN_W)
            return 23; /* CAM_POS */
        x += TV_BTN_W + 2;
        if (px >= x && px < x + TV_BTN_W)
            return 24; /* CAM_ANGLE */
        x += TV_BTN_W + 2;
        if (px >= x && px < x + TV_BTN_W)
            return 25; /* BACKGROUND */
    }
    if (py >= TV_ROW5_Y && py < TV_ROW5_Y + TV_ROW_H)
    {
        x = 2;
        if (px >= x && px < x + TV_BTN_W)
            return 26; /* RESET_MODEL */
        x += TV_BTN_W + 2;
        if (px >= x && px < x + TV_BTN_W)
            return 27; /* POSITION */
        x += TV_BTN_W + 2;
        if (px >= x && px < x + TV_BTN_W)
            return 28; /* ANGLE */
    }
    return 0;
}

/*---------------------------------------------------------------------------------
 * Numpad overlay: Back (left arrow) top-left, number display, 3x3 digits, bottom row backspace 0 OK
 *--------------------------------------------------------------------------------*/
#define NP_CANCEL_X0 2
#define NP_CANCEL_Y0 2
#define NP_CANCEL_X1 26
#define NP_CANCEL_Y1 22
#define NP_DISP_Y0 8
#define NP_DISP_Y1 28
#define NP_GRID_X0 49
#define NP_GRID_Y0 36
#define NP_BTN_W 50
#define NP_BTN_H 36
#define NP_GAP 4
#define NP_ROW_W (3 * NP_BTN_W + 2 * NP_GAP)
#define NP_BOT_Y 156
#define NP_BOT_BW 36
#define NP_BOT_GAP 4
#define NP_HIT_PERIOD 18
#define NP_HIT_SIGN 19
#define NP_HIT_CLEAR 20
#define NP_BOT_BW_SMALL 30
#define NP_BOT6_W (6 * NP_BOT_BW_SMALL + 5 * NP_BOT_GAP)
#define NP_BOT5_W (5 * NP_BOT_BW + 4 * NP_BOT_GAP)
#define NP_BOT4_W (4 * NP_BOT_BW + 3 * NP_BOT_GAP)
#define NP_TOGGLE_X0 (SUB_BMP_WIDTH - 2 - 38)
#define NP_TOGGLE_Y0 2
#define NP_TOGGLE_X1 (SUB_BMP_WIDTH - 2)
#define NP_TOGGLE_Y1 22
#define NP_REG_BTN_W 22
#define NP_REG_BTN_H 20
#define NP_REG_COLS 7     /* A-Z in register selector: 26 in 7 cols = 4 rows */
#define NP_EXT_REG_COLS 7 /* numpad reg mode: 35 in 7 cols = 5 rows */
#define NP_EXT_GRID_W (NP_EXT_REG_COLS * NP_REG_BTN_W + (NP_EXT_REG_COLS - 1) * NP_GAP)
#define NP_EXT_GRID_X0 ((SUB_BMP_WIDTH - NP_EXT_GRID_W) / 2)

static void drawNumpadScreen(void)
{
    int i, x, y;
    dmaFillHalfWords(COLOR_BG, subBuffer, SUB_BMP_WIDTH * SUB_VIS_HEIGHT * sizeof(u16));

    drawButton(NP_CANCEL_X0, NP_CANCEL_Y0, NP_CANCEL_X1, NP_CANCEL_Y1);
    {
        int cx = (NP_CANCEL_X0 + NP_CANCEL_X1) / 2, cy = (NP_CANCEL_Y0 + NP_CANCEL_Y1) / 2;
        drawTriangle(cx - 6, cy, cx + 4, cy - 6, cx + 4, cy + 6, COLOR_ARROW);
    }

    drawButton(NP_TOGGLE_X0, NP_TOGGLE_Y0, NP_TOGGLE_X1, NP_TOGGLE_Y1);
    drawStringCentered(NP_TOGGLE_X0, NP_TOGGLE_Y0, NP_TOGGLE_X1, NP_TOGGLE_Y1, numpadMode ? "123" : "REG", COLOR_TEXT);

    drawRect(NP_GRID_X0, NP_DISP_Y0, NP_GRID_X0 + NP_ROW_W, NP_DISP_Y1, COLOR_BTN);
    if (numpadMode)
    {
        int useReg = 0, r = 0;
        if (editingParamLine >= 0 && editingParamLine < scriptLines)
        {
            if (editingParamIndex == 1)
                useReg = scriptParamIsReg2[editingParamLine], r = scriptParamReg2[editingParamLine];
            else if (editingParamIndex == 2)
                useReg = scriptParamIsReg3[editingParamLine], r = scriptParamReg3[editingParamLine];
            else if (editingParamIndex == 3)
                useReg = scriptParamIsReg4[editingParamLine], r = scriptParamReg4[editingParamLine];
            else
                useReg = scriptParamIsReg[editingParamLine], r = scriptParamReg[editingParamLine];
        }
        if (useReg)
        {
            if (r < 0 || r >= NUM_EXTENDED_REGISTERS)
                r = 0;
            {
                char lbl[8];
                getRegisterLabel(r, lbl, sizeof(lbl));
                drawString(NP_GRID_X0 + 4, NP_DISP_Y0 + 2, lbl, COLOR_TEXT);
            }
        }
        else
            drawString(NP_GRID_X0 + 4, NP_DISP_Y0 + 2, "A", COLOR_TEXT);
    }
    else
    {
        if (numpadBuffer[0])
            drawString(NP_GRID_X0 + 4, NP_DISP_Y0 + 2, numpadBuffer, COLOR_TEXT);
        else
            drawString(NP_GRID_X0 + 4, NP_DISP_Y0 + 2, "0", COLOR_TEXT);
    }

    if (numpadMode)
    {
        for (i = 0; i < NUM_EXTENDED_REGISTERS; i++)
        {
            int row = i / NP_EXT_REG_COLS, col = i % NP_EXT_REG_COLS;
            x = NP_EXT_GRID_X0 + col * (NP_REG_BTN_W + NP_GAP);
            y = NP_GRID_Y0 + row * (NP_REG_BTN_H + NP_GAP);
            drawButton(x, y, x + NP_REG_BTN_W, y + NP_REG_BTN_H);
            {
                char lbl[8];
                getRegisterLabel(i, lbl, sizeof(lbl));
                drawStringCentered(x, y, x + NP_REG_BTN_W, y + NP_REG_BTN_H, lbl, COLOR_TEXT);
            }
        }
        x = (SUB_BMP_WIDTH - NP_BOT_BW) / 2;
        y = NP_BOT_Y;
        drawButton(x, y, x + NP_BOT_BW, SUB_VIS_HEIGHT - 2);
        drawStringCentered(x, y, x + NP_BOT_BW, SUB_VIS_HEIGHT - 2, "OK", COLOR_TEXT);
    }
    else
    {
        for (i = 1; i <= 9; i++)
        {
            int row = (i - 1) / 3, col = (i - 1) % 3;
            x = NP_GRID_X0 + col * (NP_BTN_W + NP_GAP);
            y = NP_GRID_Y0 + row * (NP_BTN_H + NP_GAP);
            drawButton(x, y, x + NP_BTN_W, y + NP_BTN_H);
            {
                char digit[2] = {'0' + (char)i, '\0'};
                drawStringCentered(x, y, x + NP_BTN_W, y + NP_BTN_H, digit, COLOR_TEXT);
            }
        }
        /* Bottom row (6 buttons): backspace, +/-, ., 0, C (clear), OK */
        x = (SUB_BMP_WIDTH - NP_BOT6_W) / 2;
        y = NP_BOT_Y;
        drawButton(x, y, x + NP_BOT_BW_SMALL, SUB_VIS_HEIGHT - 2);
        {
            int cx = x + NP_BOT_BW_SMALL / 2, cy = y + (SUB_VIS_HEIGHT - 2 - y) / 2;
            drawTriangle(cx - 8, cy, cx + 6, cy - 8, cx + 6, cy + 8, COLOR_ARROW);
        }
        x += NP_BOT_BW_SMALL + NP_BOT_GAP;
        drawButton(x, y, x + NP_BOT_BW_SMALL, SUB_VIS_HEIGHT - 2);
        drawStringCentered(x, y, x + NP_BOT_BW_SMALL, SUB_VIS_HEIGHT - 2, "+-", COLOR_TEXT);
        x += NP_BOT_BW_SMALL + NP_BOT_GAP;
        drawButton(x, y, x + NP_BOT_BW_SMALL, SUB_VIS_HEIGHT - 2);
        drawStringCentered(x, y, x + NP_BOT_BW_SMALL, SUB_VIS_HEIGHT - 2, ".", COLOR_TEXT);
        x += NP_BOT_BW_SMALL + NP_BOT_GAP;
        drawButton(x, y, x + NP_BOT_BW_SMALL, SUB_VIS_HEIGHT - 2);
        drawStringCentered(x, y, x + NP_BOT_BW_SMALL, SUB_VIS_HEIGHT - 2, "0", COLOR_TEXT);
        x += NP_BOT_BW_SMALL + NP_BOT_GAP;
        drawButton(x, y, x + NP_BOT_BW_SMALL, SUB_VIS_HEIGHT - 2);
        drawStringCentered(x, y, x + NP_BOT_BW_SMALL, SUB_VIS_HEIGHT - 2, "C", COLOR_TEXT);
        x += NP_BOT_BW_SMALL + NP_BOT_GAP;
        drawButton(x, y, x + NP_BOT_BW_SMALL, SUB_VIS_HEIGHT - 2);
        drawStringCentered(x, y, x + NP_BOT_BW_SMALL, SUB_VIS_HEIGHT - 2, "OK", COLOR_TEXT);
    }
}

/* Numpad hit: 0=none, 1=back, 2=OK, 3=backspace, 4=zero, 5-13=digits 1-9, 14=toggle, NP_HIT_CLEAR=clear, 16+ = registers, NP_HIT_PERIOD=period, NP_HIT_SIGN=+/-. */
static int numpadHitTest(int px, int py)
{
    int i, x, y, row, col;
    if (py >= NP_CANCEL_Y0 && py < NP_CANCEL_Y1 && px >= NP_CANCEL_X0 && px < NP_CANCEL_X1)
        return 1;
    if (py >= NP_TOGGLE_Y0 && py < NP_TOGGLE_Y1 && px >= NP_TOGGLE_X0 && px < NP_TOGGLE_X1)
        return 14; /* toggle */
    if (numpadMode)
    {
        for (i = 0; i < NUM_EXTENDED_REGISTERS; i++)
        {
            row = i / NP_EXT_REG_COLS;
            col = i % NP_EXT_REG_COLS;
            x = NP_EXT_GRID_X0 + col * (NP_REG_BTN_W + NP_GAP);
            y = NP_GRID_Y0 + row * (NP_REG_BTN_H + NP_GAP);
            if (px >= x && px < x + NP_REG_BTN_W && py >= y && py < y + NP_REG_BTN_H)
                return 16 + i;
        }
        x = (SUB_BMP_WIDTH - NP_BOT_BW) / 2;
        y = NP_BOT_Y;
        if (py >= y && py < SUB_VIS_HEIGHT - 2 && px >= x && px < x + NP_BOT_BW)
            return 2; /* OK */
    }
    else
    {
        /* Bottom row (6 buttons): backspace, +/-, period, 0, C, OK */
        if (py >= NP_BOT_Y && py < SUB_VIS_HEIGHT)
        {
            x = (SUB_BMP_WIDTH - NP_BOT6_W) / 2;
            if (px >= x && px < x + NP_BOT_BW_SMALL)
                return 3; /* backspace */
            x += NP_BOT_BW_SMALL + NP_BOT_GAP;
            if (px >= x && px < x + NP_BOT_BW_SMALL)
                return NP_HIT_SIGN; /* +/- */
            x += NP_BOT_BW_SMALL + NP_BOT_GAP;
            if (px >= x && px < x + NP_BOT_BW_SMALL)
                return NP_HIT_PERIOD; /* period (decimal point) */
            x += NP_BOT_BW_SMALL + NP_BOT_GAP;
            if (px >= x && px < x + NP_BOT_BW_SMALL)
                return 4; /* 0 */
            x += NP_BOT_BW_SMALL + NP_BOT_GAP;
            if (px >= x && px < x + NP_BOT_BW_SMALL)
                return NP_HIT_CLEAR; /* clear */
            x += NP_BOT_BW_SMALL + NP_BOT_GAP;
            if (px >= x && px < x + NP_BOT_BW_SMALL)
                return 2; /* OK */
        }
        for (i = 1; i <= 9; i++)
        {
            row = (i - 1) / 3;
            col = (i - 1) % 3;
            x = NP_GRID_X0 + col * (NP_BTN_W + NP_GAP);
            y = NP_GRID_Y0 + row * (NP_BTN_H + NP_GAP);
            if (px >= x && px < x + NP_BTN_W && py >= y && py < y + NP_BTN_H)
                return 4 + i; /* 5..13 for 1..9 */
        }
    }
    return 0;
}

/*---------------------------------------------------------------------------------
 * Register selector: same compact layout as numpad REG mode (back, small display, 4x2 A-H, OK)
 *--------------------------------------------------------------------------------*/
#define RS_BACK_X0 2
#define RS_BACK_Y0 2
#define RS_BACK_X1 26
#define RS_BACK_Y1 22
#define RS_DISP_Y0 8
#define RS_DISP_Y1 28
#define RS_GRID_W (NP_REG_COLS * NP_REG_BTN_W + (NP_REG_COLS - 1) * NP_GAP)
#define RS_GRID_X0 ((SUB_BMP_WIDTH - RS_GRID_W) / 2)
#define RS_GRID_Y0 36

static void drawRegisterSelectScreen(void)
{
    int i, x, y, row, col;
    int r = 0;
    dmaFillHalfWords(COLOR_BG, subBuffer, SUB_BMP_WIDTH * SUB_VIS_HEIGHT * sizeof(u16));

    drawButton(RS_BACK_X0, RS_BACK_Y0, RS_BACK_X1, RS_BACK_Y1);
    {
        int cx = (RS_BACK_X0 + RS_BACK_X1) / 2, cy = (RS_BACK_Y0 + RS_BACK_Y1) / 2;
        drawTriangle(cx - 6, cy, cx + 4, cy - 6, cx + 4, cy + 6, COLOR_ARROW);
    }

    if (editingParamLine >= 0 && editingParamLine < scriptLines)
    {
        r = scriptReg[editingParamLine];
        if (r < 0)
            r = 0;
        if (r >= NUM_REGISTERS)
            r = NUM_REGISTERS - 1;
    }
    drawRect(RS_GRID_X0, RS_DISP_Y0, RS_GRID_X0 + RS_GRID_W, RS_DISP_Y1, COLOR_BTN);
    {
        char letter[8];
        getRegisterLabel(r, letter, sizeof(letter));
        drawStringCentered(RS_GRID_X0, RS_DISP_Y0, RS_GRID_X0 + RS_GRID_W, RS_DISP_Y1, letter, COLOR_TEXT);
    }

    for (i = 0; i < NUM_REGISTERS; i++)
    {
        row = i / NP_REG_COLS;
        col = i % NP_REG_COLS;
        x = RS_GRID_X0 + col * (NP_REG_BTN_W + NP_GAP);
        y = RS_GRID_Y0 + row * (NP_REG_BTN_H + NP_GAP);
        drawButton(x, y, x + NP_REG_BTN_W, y + NP_REG_BTN_H);
        {
            char letter[2] = {'A' + (char)i, '\0'};
            drawStringCentered(x, y, x + NP_REG_BTN_W, y + NP_REG_BTN_H, letter, COLOR_TEXT);
        }
    }

    x = (SUB_BMP_WIDTH - NP_BOT_BW) / 2;
    y = NP_BOT_Y;
    drawButton(x, y, x + NP_BOT_BW, SUB_VIS_HEIGHT - 2);
    drawStringCentered(x, y, x + NP_BOT_BW, SUB_VIS_HEIGHT - 2, "OK", COLOR_TEXT);
}

/* Register select hit: 0=none, 1=back, 2=OK, 3-28=letters A-Z. */
static int registerSelectHitTest(int px, int py)
{
    int i, x, y, row, col;
    if (py >= RS_BACK_Y0 && py < RS_BACK_Y1 && px >= RS_BACK_X0 && px < RS_BACK_X1)
        return 1;
    x = (SUB_BMP_WIDTH - NP_BOT_BW) / 2;
    y = NP_BOT_Y;
    if (py >= y && py < SUB_VIS_HEIGHT - 2 && px >= x && px < x + NP_BOT_BW)
        return 2;
    for (i = 0; i < NUM_REGISTERS; i++)
    {
        row = i / NP_REG_COLS;
        col = i % NP_REG_COLS;
        x = RS_GRID_X0 + col * (NP_REG_BTN_W + NP_GAP);
        y = RS_GRID_Y0 + row * (NP_REG_BTN_H + NP_GAP);
        if (px >= x && px < x + NP_REG_BTN_W && py >= y && py < y + NP_REG_BTN_H)
            return 3 + i;
    }
    return 0;
}

/* X offset where sleep param number starts (after "SLEEP "). */
#define SCRIPT_TEXT_X 6
#define SLEEP_LABEL_WIDTH (stringWidth("SLEEP ") + (FONT_W + 1))

/* Hit-test: return 1-3=top row, 4-7=nav (up, down, token, clear), -3/-2/-1=script area, 0=none.
 * setSelectedLine: 1 = update selectedLine when touching script area (use only on touch-start / kDown). */
static int hitTest(int px, int py, int setSelectedLine)
{
    int x;
    if (py >= TOP_Y0 && py < TOP_Y1)
    {
        if (px >= 2 && px < 2 + BTN_W)
            return 1;
        if (px >= STOP_BTN_X0 && px < STOP_BTN_X1)
            return 2;
        if (px >= STEP_BTN_X0 && px < STEP_BTN_X1)
            return 3;
        if (px >= SCRIPTS_BTN_X0 && px < SCRIPTS_BTN_X1)
            return 8;
    }
    if (py >= NAV_Y0 && py < NAV_Y1)
    {
        x = 2;
        if (px >= x && px < x + NAV_BTN_W)
            return 4;
        x += NAV_BTN_W + 2;
        if (px >= x && px < x + NAV_BTN_W)
            return 5;
        x += NAV_BTN_W + 2;
        if (px >= x && px < x + NAV_BTN_W)
            return 6;
        x += NAV_BTN_W + 2;
        if (px >= x && px < x + NAV_BTN_W)
            return 7;
    }
    /* Script area: set selected line only on touch-start (setSelectedLine); tap on param opens numpad or register selector. lineIdx -1 = blank row above first line. */
    if (py >= SCRIPT_Y0 && py < SCRIPT_Y1)
    {
        int row = (py - SCRIPT_Y0) / LINE_H;
        int lineIdx = scriptScrollOffset + row;
        if (lineIdx >= -1 && lineIdx < scriptLines)
        {
            if (setSelectedLine)
                selectedLine = lineIdx;
            if (lineIdx >= 0 && !scriptRunning)
            {
                if (tokenEquals(script[lineIdx], "model") && px >= SCRIPT_TEXT_X + stringWidth("MODEL ") + (FONT_W + 1))
                {
                    editingParamLine = lineIdx;
                    editingParamIndex = 0;
                    return -2;
                }
                if (tokenEquals(script[lineIdx], "sleep") && px >= SCRIPT_TEXT_X + SLEEP_LABEL_WIDTH)
                {
                    editingParamLine = lineIdx;
                    editingParamIndex = 0;
                    return -2; /* open numpad */
                }
                if (tokenEquals(script[lineIdx], "next_color") && px >= SCRIPT_TEXT_X + stringWidth("NEXT_COLOR ") + (FONT_W + 1))
                {
                    editingParamLine = lineIdx;
                    editingParamIndex = 0;
                    return -2;
                }
                if (tokenEquals(script[lineIdx], "rotate"))
                {
                    int slotW = (FONT_W + 1) * 6;
                    int tx0 = SCRIPT_TEXT_X + stringWidth("ROTATE ") + (FONT_W + 1);
                    if (px >= tx0 && px < tx0 + slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 0;
                        return -2;
                    }
                    if (px >= tx0 + slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 1;
                        return -2;
                    }
                }
                if (tokenEquals(script[lineIdx], "translate"))
                {
                    int tx0 = SCRIPT_TEXT_X + stringWidth("TRANSLATE ") + (FONT_W + 1);
                    int slotW = (FONT_W + 1) * 6;
                    if (px >= tx0 && px < tx0 + slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 0;
                        return -2;
                    }
                    if (px >= tx0 + slotW && px < tx0 + 2 * slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 1;
                        return -2;
                    }
                    if (px >= tx0 + 2 * slotW && px < tx0 + 3 * slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 2;
                        return -2;
                    }
                    if (px >= tx0 + 3 * slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 3;
                        return -2;
                    }
                }
                if (tokenEquals(script[lineIdx], "position"))
                {
                    int tx0 = SCRIPT_TEXT_X + stringWidth("POSITION ") + (FONT_W + 1);
                    int slotW = (FONT_W + 1) * 6;
                    if (px >= tx0 && px < tx0 + slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 0;
                        return -2;
                    }
                    if (px >= tx0 + slotW && px < tx0 + 2 * slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 1;
                        return -2;
                    }
                    if (px >= tx0 + 2 * slotW && px < tx0 + 3 * slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 2;
                        return -2;
                    }
                    if (px >= tx0 + 3 * slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 3;
                        return -2;
                    }
                }
                if (tokenEquals(script[lineIdx], "angle"))
                {
                    int slotW = (FONT_W + 1) * 6;
                    int tx0 = SCRIPT_TEXT_X + stringWidth("ANGLE ") + (FONT_W + 1);
                    if (px >= tx0 && px < tx0 + slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 0;
                        return -2;
                    }
                    if (px >= tx0 + slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 1;
                        return -2;
                    }
                }
                if (tokenEquals(script[lineIdx], "cam_pos"))
                {
                    int slotW = (FONT_W + 1) * 6;
                    int tx0 = SCRIPT_TEXT_X + stringWidth("CAM_POS ") + (FONT_W + 1);
                    if (px >= tx0 && px < tx0 + slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 0;
                        return -2;
                    }
                    if (px >= tx0 + slotW && px < tx0 + 2 * slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 1;
                        return -2;
                    }
                    if (px >= tx0 + 2 * slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 2;
                        return -2;
                    }
                }
                if (tokenEquals(script[lineIdx], "cam_angle"))
                {
                    int slotW = (FONT_W + 1) * 6;
                    int tx0 = SCRIPT_TEXT_X + stringWidth("CAM_ANGLE ") + (FONT_W + 1);
                    if (px >= tx0 && px < tx0 + slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 0;
                        return -2;
                    }
                    if (px >= tx0 + slotW)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 1;
                        return -2;
                    }
                }
                if (tokenEquals(script[lineIdx], "background") && px >= SCRIPT_TEXT_X + stringWidth("BACKGROUND ") + (FONT_W + 1))
                {
                    editingParamLine = lineIdx;
                    editingParamIndex = 0;
                    return -2;
                }
                if (tokenEquals(script[lineIdx], "reset_model") && px >= SCRIPT_TEXT_X + stringWidth("RESET_MODEL ") + (FONT_W + 1))
                {
                    editingParamLine = lineIdx;
                    editingParamIndex = 0;
                    return -2;
                }
                if (tokenEquals(script[lineIdx], "if_true"))
                {
                    int tx0 = SCRIPT_TEXT_X + stringWidth("IF_TRUE ") + (FONT_W + 1);
                    if (px >= tx0)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 0;
                        return -2;
                    }
                }
                if (hasTwoNumberParams(script[lineIdx]))
                {
                    int preW = tokenEquals(script[lineIdx], "if_gt") ? stringWidth("IF_GT ") : stringWidth("IF_LT ");
                    int reg0End = SCRIPT_TEXT_X + preW + (FONT_W + 1) * 4;
                    int reg1Start = reg0End + (FONT_W + 1) * 2;
                    if (px >= SCRIPT_TEXT_X + preW && px < reg0End)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 0;
                        return -2;
                    }
                    if (px >= reg1Start)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 1;
                        return -2;
                    }
                }
                if (isRegToken(script[lineIdx]))
                {
                    int regW = (FONT_W + 1);
                    int preW;
                    if (tokenEquals(script[lineIdx], "set"))
                        preW = stringWidth("SET ");
                    else if (tokenEquals(script[lineIdx], "add"))
                        preW = stringWidth("ADD ");
                    else if (tokenEquals(script[lineIdx], "multiply"))
                        preW = stringWidth("MULTIPLY ");
                    else
                        preW = stringWidth("SUBTRACT ");
                    int regStart = SCRIPT_TEXT_X + preW;
                    int regEnd = regStart + regW;
                    int numStart = regEnd + regW + regW; /* letter + space */
                    if (px >= regStart && px < regEnd)
                    {
                        editingParamLine = lineIdx;
                        return -3; /* open register selector */
                    }
                    if (px >= numStart)
                    {
                        editingParamLine = lineIdx;
                        editingParamIndex = 0;
                        return -2; /* open numpad */
                    }
                }
            }
        }
        return -1; /* consumed but no button */
    }
    return 0;
}

/* Find line index of matching END_IF after scriptIP (skip nested IF_GT/IF_LT/IF_TRUE blocks). Returns -1 if not found. */
static int findMatchingEndIf(void)
{
    int depth = 0;
    int i;
    for (i = scriptIP + 1; i < scriptLines; i++)
    {
        if (tokenEquals(script[i], "if_gt") || tokenEquals(script[i], "if_lt") || tokenEquals(script[i], "if_true"))
            depth++;
        else if (tokenEquals(script[i], "end_if"))
        {
            if (depth == 0)
                return i;
            depth--;
        }
    }
    return -1;
}

/* Execute one instruction; return 1 if advanced. stepMode: 1 = one Step click finishes current sleep in one tick. Uses per-model state. */
static int scriptStep(int stepMode)
{
    if (scriptIP < 0 || scriptIP >= scriptLines)
        return 0;
    if (sleepFramesLeft > 0)
    {
        if (stepMode)
            sleepFramesLeft = 0; /* Step button: finish sleep in one tick */
        else
            sleepFramesLeft--;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "model"))
    {
        int idx = (int)getNumberParamValue(scriptIP, 0);
        if (idx >= 0 && idx < MAX_MODELS)
        {
            modelActive[idx] = 1;
            modelAngle[idx] = 0.0f;
            modelX[idx] = modelY[idx] = modelZ[idx] = 0.0f;
            modelColorIndex[idx] = 0;
        }
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "reset_model"))
    {
        int idx = (int)getNumberParamValue(scriptIP, 0);
        if (idx >= 0 && idx < MAX_MODELS && modelActive[idx])
        {
            modelAngle[idx] = 0.0f;
            modelX[idx] = modelY[idx] = modelZ[idx] = 0.0f;
        }
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "next_color"))
    {
        int idx = (int)getNumberParamValue(scriptIP, 0);
        if (idx >= 0 && idx < MAX_MODELS && modelActive[idx])
            modelColorIndex[idx] = (modelColorIndex[idx] + 1) % 3;
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "beep"))
    {
        soundEnable();
        if (beepFramesLeft > 0)
            soundPause(beepChannel);
        /* Pong bounce: short mid-high tone, ~0.1s */
        beepChannel = soundPlayPSG(DutyCycle_50, 2800, 127, 64);
        beepFramesLeft = 6; /* ~0.1s at 60fps */
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "cam_pos"))
    {
        camX = getNumberParamValue(scriptIP, 0);
        camY = getNumberParamValue(scriptIP, 1);
        camZ = getNumberParamValue(scriptIP, 2);
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "cam_angle"))
    {
        camYaw = getNumberParamValue(scriptIP, 0);
        camPitch = getNumberParamValue(scriptIP, 1);
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "background"))
    {
        int idx = (int)getNumberParamValue(scriptIP, 0);
        if (idx < 0)
            idx = 0;
        if (idx > 3)
            idx = 3;
        bgColorIndex = idx;
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "sleep"))
    {
        {
            float sec = getNumberParamValue(scriptIP, 0);
            if (sec < 0.016f)
                sec = 0.016f;
            sleepFramesLeft = (int)(sec * 60.0f);
            if (sleepFramesLeft < 1)
                sleepFramesLeft = 1;
        }
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "set"))
    {
        int r = scriptReg[scriptIP];
        if (r >= 0 && r < NUM_REGISTERS)
            registers[r] = getNumberParamValue(scriptIP, 0);
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "add"))
    {
        int r = scriptReg[scriptIP];
        if (r >= 0 && r < NUM_REGISTERS)
            registers[r] += getNumberParamValue(scriptIP, 0);
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "subtract"))
    {
        int r = scriptReg[scriptIP];
        if (r >= 0 && r < NUM_REGISTERS)
            registers[r] -= getNumberParamValue(scriptIP, 0);
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "rotate"))
    {
        int idx = (int)getNumberParamValue(scriptIP, 0);
        float amount = getNumberParamValue(scriptIP, 1);
        if (idx >= 0 && idx < MAX_MODELS && modelActive[idx])
            modelAngle[idx] += amount;
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "multiply"))
    {
        int r = scriptReg[scriptIP];
        if (r >= 0 && r < NUM_REGISTERS)
            registers[r] *= getNumberParamValue(scriptIP, 0);
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "translate"))
    {
        int idx = (int)getNumberParamValue(scriptIP, 0);
        if (idx >= 0 && idx < MAX_MODELS && modelActive[idx])
        {
            modelX[idx] += getNumberParamValue(scriptIP, 1);
            modelY[idx] += getNumberParamValue(scriptIP, 2);
            modelZ[idx] += getNumberParamValue(scriptIP, 3);
        }
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "position"))
    {
        int idx = (int)getNumberParamValue(scriptIP, 0);
        if (idx >= 0 && idx < MAX_MODELS && modelActive[idx])
        {
            modelX[idx] = getNumberParamValue(scriptIP, 1);
            modelY[idx] = getNumberParamValue(scriptIP, 2);
            modelZ[idx] = getNumberParamValue(scriptIP, 3);
        }
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "angle"))
    {
        int idx = (int)getNumberParamValue(scriptIP, 0);
        float amount = getNumberParamValue(scriptIP, 1);
        if (idx >= 0 && idx < MAX_MODELS && modelActive[idx])
            modelAngle[idx] = amount;
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "if_gt"))
    {
        float left = getNumberParamValue(scriptIP, 0);
        float right = getNumberParamValue(scriptIP, 1);
        if (left > right)
            scriptIP++;
        else
        {
            int endIf = findMatchingEndIf();
            scriptIP = (endIf >= 0 ? endIf + 1 : scriptLines);
        }
        return 1;
    }
    if (tokenEquals(script[scriptIP], "if_lt"))
    {
        float left = getNumberParamValue(scriptIP, 0);
        float right = getNumberParamValue(scriptIP, 1);
        if (left < right)
            scriptIP++;
        else
        {
            int endIf = findMatchingEndIf();
            scriptIP = (endIf >= 0 ? endIf + 1 : scriptLines);
        }
        return 1;
    }
    if (tokenEquals(script[scriptIP], "if_true"))
    {
        float v = getNumberParamValue(scriptIP, 0);
        if (v != 0.0f)
            scriptIP++;
        else
        {
            int endIf = findMatchingEndIf();
            scriptIP = (endIf >= 0 ? endIf + 1 : scriptLines);
        }
        return 1;
    }
    if (tokenEquals(script[scriptIP], "end_if"))
    {
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "loop"))
    {
        if (loopSp < LOOP_STACK_MAX)
            loopStack[loopSp++] = scriptIP;
        scriptIP++;
        return 1;
    }
    if (tokenEquals(script[scriptIP], "end_loop"))
    {
        if (loopSp > 0)
            scriptIP = loopStack[loopSp - 1] + 1; /* jump to first line after loop */
        else
            scriptIP++;
        return 1;
    }
    scriptIP++;
    return 1;
}

/*---------------------------------------------------------------------------------
 * Draw solid-colored cube. Wireframe is drawn separately and slightly larger.
 *--------------------------------------------------------------------------------*/
static void drawCube(u16 colorRgb15)
{
    int r = (colorRgb15 >> 0) & 0x1F;
    int g = (colorRgb15 >> 5) & 0x1F;
    int b = (colorRgb15 >> 10) & 0x1F;
    glColor3b(r * 255 / 31, g * 255 / 31, b * 255 / 31);

    glBegin(GL_QUADS);
    /* +Z */
    glVertex3f(-1.0f, 1.0f, 1.0f);
    glVertex3f(1.0f, 1.0f, 1.0f);
    glVertex3f(1.0f, -1.0f, 1.0f);
    glVertex3f(-1.0f, -1.0f, 1.0f);
    /* -Z */
    glVertex3f(-1.0f, 1.0f, -1.0f);
    glVertex3f(-1.0f, -1.0f, -1.0f);
    glVertex3f(1.0f, -1.0f, -1.0f);
    glVertex3f(1.0f, 1.0f, -1.0f);
    /* +Y top */
    glVertex3f(-1.0f, 1.0f, 1.0f);
    glVertex3f(-1.0f, 1.0f, -1.0f);
    glVertex3f(1.0f, 1.0f, -1.0f);
    glVertex3f(1.0f, 1.0f, 1.0f);
    /* -Y bottom */
    glVertex3f(-1.0f, -1.0f, 1.0f);
    glVertex3f(1.0f, -1.0f, 1.0f);
    glVertex3f(1.0f, -1.0f, -1.0f);
    glVertex3f(-1.0f, -1.0f, -1.0f);
    /* +X right */
    glVertex3f(1.0f, 1.0f, 1.0f);
    glVertex3f(1.0f, 1.0f, -1.0f);
    glVertex3f(1.0f, -1.0f, -1.0f);
    glVertex3f(1.0f, -1.0f, 1.0f);
    /* -X left */
    glVertex3f(-1.0f, 1.0f, 1.0f);
    glVertex3f(-1.0f, -1.0f, 1.0f);
    glVertex3f(-1.0f, -1.0f, -1.0f);
    glVertex3f(-1.0f, 1.0f, -1.0f);
    glEnd();
}

/* Wireframe: drawn slightly larger than cube so it's visible (scale and line thickness) */
#define WIREFRAME_SCALE 1.06f
#define WIREFRAME_THICK 0.045f

/* Draw one thin quad along edge A->B with perpendicular (px,py,pz). */
static void drawEdgeQuad(float ax, float ay, float az, float bx, float by, float bz,
                         float px, float py, float pz)
{
    float t = WIREFRAME_THICK;
    glVertex3f(ax + t * px, ay + t * py, az + t * pz);
    glVertex3f(bx + t * px, by + t * py, bz + t * pz);
    glVertex3f(bx - t * px, by - t * py, bz - t * pz);
    glVertex3f(ax - t * px, ay - t * py, az - t * pz);
}

/* Draw one edge as two perpendicular thin quads so it's visible from any angle. */
static void drawEdge(float ax, float ay, float az, float bx, float by, float bz)
{
    float dx = bx - ax, dy = by - ay, dz = bz - az;
    float p1x, p1y, p1z, p2x, p2y, p2z;
    if (dx * dx >= dy * dy && dx * dx >= dz * dz)
    {
        p1x = 0;
        p1y = 1;
        p1z = 0;
        p2x = 0;
        p2y = 0;
        p2z = 1;
    }
    else if (dy * dy >= dz * dz)
    {
        p1x = 1;
        p1y = 0;
        p1z = 0;
        p2x = 0;
        p2y = 0;
        p2z = 1;
    }
    else
    {
        p1x = 1;
        p1y = 0;
        p1z = 0;
        p2x = 0;
        p2y = 1;
        p2z = 0;
    }
    drawEdgeQuad(ax, ay, az, bx, by, bz, p1x, p1y, p1z);
    drawEdgeQuad(ax, ay, az, bx, by, bz, p2x, p2y, p2z);
}

/* Draw black wireframe cube (12 edges as thin quads), scaled slightly larger than unit cube. */
static void drawWireframeCube(void)
{
    float s = WIREFRAME_SCALE;
    /* Disable culling so all edge quads are visible from any angle */
    glPolyFmt(POLY_ALPHA(31) | POLY_CULL_NONE);
    glColor3b(0, 0, 0);
    glBegin(GL_QUADS);
    /* Front face +Z */
    drawEdge(-s, s, s, s, s, s);
    drawEdge(s, s, s, s, -s, s);
    drawEdge(s, -s, s, -s, -s, s);
    drawEdge(-s, -s, s, -s, s, s);
    /* Back face -Z */
    drawEdge(-s, s, -s, s, s, -s);
    drawEdge(s, s, -s, s, -s, -s);
    drawEdge(s, -s, -s, -s, -s, -s);
    drawEdge(-s, -s, -s, -s, s, -s);
    /* Connecting edges */
    drawEdge(-s, s, s, -s, s, -s);
    drawEdge(s, s, s, s, s, -s);
    drawEdge(s, -s, s, s, -s, -s);
    drawEdge(-s, -s, s, -s, -s, -s);
    glEnd();
    /* Restore back-face culling for filled geometry */
    glPolyFmt(POLY_ALPHA(31) | POLY_CULL_BACK);
}

/*---------------------------------------------------------------------------------
 * Init 3D on main (top) screen.
 *--------------------------------------------------------------------------------*/
static void init3D(void)
{
    glInit();
    glEnable(GL_ANTIALIAS);
    glClearColor(0, 0, 0, 31);
    glClearPolyID(63);
    glClearDepth(0x7FFF);
    glViewport(0, 0, 255, 191);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(70, 256.0f / 192.0f, 0.1f, 100.0f);
    glMatrixMode(GL_MODELVIEW);
    glPolyFmt(POLY_ALPHA(31) | POLY_CULL_BACK);
}

int main(void)
{
    /* Top = 3D, bottom = 2D */
    lcdMainOnTop();
    powerOn(POWER_LCD | POWER_MATRIX);

    videoSetMode(MODE_0_3D);
    videoSetModeSub(MODE_5_2D | DISPLAY_BG3_ACTIVE);

    vramSetBankC(VRAM_C_SUB_BG);
    subBgId = bgInitSub(3, BgType_Bmp16, BgSize_B16_256x256, 0, 0);
    subBuffer = (u16 *)bgGetGfxPtr(subBgId);

    init3D();
    scriptLoadDefault();
    scriptSaveToSlot(0); /* script 1 gets default (Pong demo) */
    currentScriptSlot = 0;
    {
        int s;
        for (s = 1; s < NUM_SCRIPT_SLOTS; s++)
            scriptSlotLines[s] = 0; /* scripts 2..6 empty */
    }
    scriptResetExecution();
    scriptActive = 0;
    elapsedTimeSeconds = 0.0f;
    numpadActive = 0;
    registerSelectActive = 0;
    tokenViewActive = 0;
    scriptMenuActive = 0;
    skipNextScriptTouchAfterTokenInsert = 0;
    scrollUpCounter = -1;
    scrollDownCounter = -1;
    stepRepeatCounter = -1;
    clearRepeatCounter = -1;

    soundEnable();
    beepFramesLeft = 0;
    camX = 0.0f;
    camY = 0.0f;
    camZ = 4.0f;
    camYaw = 0.0f;
    camPitch = 0.0f;
    bgColorIndex = 0;

    int stepRequest = 0; /* step button clicked when paused */

    while (1)
    {
        swiWaitForVBlank();
        scanKeys();
        touchPosition touch;
        touchRead(&touch);

        int kDown = keysDown();

        if (registerSelectActive)
        {
            if (kDown & KEY_TOUCH && touch.px > 0)
            {
                int rh = registerSelectHitTest(touch.px, touch.py);
                if (rh == 1 || rh == 2)
                    registerSelectActive = 0;
                else if (rh >= 3 && rh <= 3 + NUM_REGISTERS - 1 && editingParamLine >= 0 && editingParamLine < scriptLines)
                {
                    scriptReg[editingParamLine] = rh - 3; /* 0-25 = A-Z */
                    registerSelectActive = 0;             /* instant apply and close */
                }
            }
        }
        else if (numpadActive)
        {
            /* Numpad overlay: handle touch */
            if (kDown & KEY_TOUCH && touch.px > 0)
            {
                int nh = numpadHitTest(touch.px, touch.py);
                if (nh == 1)
                {
                    numpadActive = 0;
                }
                else if (nh == 14)
                {
                    numpadMode = 1 - numpadMode;
                }
                else if (nh == 2)
                {
                    /* OK: apply and close */
                    if (numpadMode)
                    {
                        /* register mode: selection already stored on letter tap */
                    }
                    else
                    {
                        float v = 0.0f;
                        if (numpadBuffer[0])
                            sscanf(numpadBuffer, "%f", &v);
                        if (editingParamLine >= 0 && editingParamLine < scriptLines)
                        {
                            if (editingParamIndex == 1)
                            {
                                scriptParam2[editingParamLine] = v;
                                scriptParamIsReg2[editingParamLine] = 0;
                            }
                            else if (editingParamIndex == 2)
                            {
                                scriptParam3[editingParamLine] = v;
                                scriptParamIsReg3[editingParamLine] = 0;
                            }
                            else if (editingParamIndex == 3)
                            {
                                scriptParam4[editingParamLine] = v;
                                scriptParamIsReg4[editingParamLine] = 0;
                            }
                            else
                            {
                                scriptParam[editingParamLine] = v;
                                scriptParamIsReg[editingParamLine] = 0;
                            }
                        }
                    }
                    numpadActive = 0;
                }
                else if (nh == NP_HIT_PERIOD && !numpadMode)
                {
                    /* decimal point (must be before nh>=16 check: 18 would else be treated as register C) */
                    {
                        int len = (int)strlen(numpadBuffer);
                        int hasPeriod = 0, j;
                        for (j = 0; j < len; j++)
                            if (numpadBuffer[j] == '.')
                                hasPeriod = 1;
                        if (!hasPeriod && len < NUMPAD_BUF_LEN - 1)
                        {
                            numpadBuffer[len] = '.';
                            numpadBuffer[len + 1] = '\0';
                        }
                    }
                }
                else if (nh == NP_HIT_SIGN && !numpadMode)
                {
                    /* +/- toggle negative (must be before nh>=16: 19 would else be register D) */
                    {
                        int len = (int)strlen(numpadBuffer);
                        if (numpadBuffer[0] == '-')
                        {
                            memmove(numpadBuffer, numpadBuffer + 1, (unsigned)(len));
                            numpadBuffer[len - 1] = '\0';
                        }
                        else if (len < NUMPAD_BUF_LEN - 1)
                        {
                            memmove(numpadBuffer + 1, numpadBuffer, (unsigned)(len + 1));
                            numpadBuffer[0] = '-';
                            numpadBuffer[len + 1] = '\0';
                        }
                    }
                }
                else if (nh == NP_HIT_CLEAR && !numpadMode)
                {
                    numpadBuffer[0] = '\0';
                }
                else if (nh >= 16 && nh <= 16 + NUM_EXTENDED_REGISTERS - 1 && editingParamLine >= 0 && editingParamLine < scriptLines)
                {
                    if (editingParamIndex == 1)
                    {
                        scriptParamIsReg2[editingParamLine] = 1;
                        scriptParamReg2[editingParamLine] = nh - 16;
                    }
                    else if (editingParamIndex == 2)
                    {
                        scriptParamIsReg3[editingParamLine] = 1;
                        scriptParamReg3[editingParamLine] = nh - 16;
                    }
                    else if (editingParamIndex == 3)
                    {
                        scriptParamIsReg4[editingParamLine] = 1;
                        scriptParamReg4[editingParamLine] = nh - 16;
                    }
                    else
                    {
                        scriptParamIsReg[editingParamLine] = 1;
                        scriptParamReg[editingParamLine] = nh - 16;
                    }
                    numpadActive = 0; /* instant apply and close */
                }
                else if (nh == 3)
                {
                    /* backspace: remove last character */
                    {
                        int len = (int)strlen(numpadBuffer);
                        if (len > 0)
                        {
                            numpadBuffer[len - 1] = '\0';
                        }
                    }
                }
                else if (nh == 4)
                {
                    int len = (int)strlen(numpadBuffer);
                    if (len < NUMPAD_BUF_LEN - 1)
                    {
                        numpadBuffer[len] = '0';
                        numpadBuffer[len + 1] = '\0';
                    }
                }
                else if (nh >= 5 && nh <= 13)
                {
                    int len = (int)strlen(numpadBuffer);
                    if (len < NUMPAD_BUF_LEN - 1)
                    {
                        numpadBuffer[len] = (char)('0' + (nh - 4));
                        numpadBuffer[len + 1] = '\0';
                    }
                }
            }
        }
        else if (tokenViewActive)
        {
            if (kDown & KEY_TOUCH && touch.px > 0)
            {
                int th = tokenViewHitTest(touch.px, touch.py);
                if (th == 1)
                    tokenViewActive = 0;
                else if (th == 7)
                    scriptInsertLine("next_color"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 8)
                    scriptInsertLine("sleep"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 9)
                    scriptInsertLine("loop"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 10)
                    scriptInsertLine("end_loop"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 11)
                    scriptInsertLine("set"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 12)
                    scriptInsertLine("add"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 13)
                    scriptInsertLine("subtract"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 14)
                    scriptInsertLine("rotate"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 15)
                    scriptInsertLine("multiply"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 16)
                    scriptInsertLine("translate"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 17)
                    scriptInsertLine("if_gt"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 18)
                    scriptInsertLine("if_lt"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 19)
                    scriptInsertLine("model"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 20)
                    scriptInsertLine("if_true"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 21)
                    scriptInsertLine("end_if"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 22)
                    scriptInsertLine("beep"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 23)
                    scriptInsertLine("cam_pos"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 24)
                    scriptInsertLine("cam_angle"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 25)
                    scriptInsertLine("background"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 26)
                    scriptInsertLine("reset_model"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 27)
                    scriptInsertLine("position"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
                else if (th == 28)
                    scriptInsertLine("angle"), tokenViewActive = 0, skipNextScriptTouchAfterTokenInsert = 1;
            }
        }
        else if (scriptMenuActive)
        {
            if (kDown & KEY_TOUCH && touch.px > 0)
            {
                int smh = scriptMenuHitTest(touch.px, touch.py);
                if (smh == 1)
                    scriptMenuActive = 0;
                else if (smh >= 2 && smh <= 7)
                {
                    int slot = smh - 2;                  /* 0..5 = script 1..6 */
                    scriptSaveToSlot(currentScriptSlot); /* save current editor to current slot */
                    scriptLoadFromSlot(slot);
                    currentScriptSlot = slot;
                    scriptResetExecution();
                    scriptActive = 0; /* switching script: not running */
                    scriptMenuActive = 0;
                }
            }
        }
        else
        {
            /* Touch: handle buttons and script area */
            if (kDown & KEY_TOUCH && touch.px > 0)
            {
                if (skipNextScriptTouchAfterTokenInsert)
                {
                    skipNextScriptTouchAfterTokenInsert = 0;
                    /* Consume touch so it doesn't run hitTest and overwrite selectedLine */
                }
                else
                {
                    int hit = hitTest(touch.px, touch.py, 1); /* 1 = touch just started (kDown), allow set selectedLine */
                    if (hit == -3)
                        registerSelectActive = 1; /* editingParamLine already set by hitTest */
                    else if (hit == -2)
                    {
                        /* Open numpad for number param (editingParamLine, editingParamIndex set) */
                        {
                            int useParam2 = (editingParamIndex == 1 && editingParamLine >= 0 && editingParamLine < scriptLines);
                            int useParam3 = (editingParamIndex == 2 && editingParamLine >= 0 && editingParamLine < scriptLines);
                            int useParam4 = (editingParamIndex == 3 && editingParamLine >= 0 && editingParamLine < scriptLines);
                            numpadMode = 0;
                            if (useParam2)
                                numpadMode = scriptParamIsReg2[editingParamLine] ? 1 : 0;
                            else if (useParam3)
                                numpadMode = scriptParamIsReg3[editingParamLine] ? 1 : 0;
                            else if (useParam4)
                                numpadMode = scriptParamIsReg4[editingParamLine] ? 1 : 0;
                            else if (editingParamLine >= 0 && editingParamLine < scriptLines)
                                numpadMode = scriptParamIsReg[editingParamLine] ? 1 : 0;
                            if (!numpadMode)
                            {
                                if (useParam2)
                                    formatSleepParam(scriptParam2[editingParamLine], numpadBuffer, NUMPAD_BUF_LEN);
                                else if (useParam3)
                                    formatSleepParam(scriptParam3[editingParamLine], numpadBuffer, NUMPAD_BUF_LEN);
                                else if (useParam4)
                                    formatSleepParam(scriptParam4[editingParamLine], numpadBuffer, NUMPAD_BUF_LEN);
                                else
                                    formatSleepParam(scriptParam[editingParamLine], numpadBuffer, NUMPAD_BUF_LEN);
                            }
                        }
                        numpadActive = 1;
                    }
                    else if (hit == 1)
                    {
                        scriptRunning = !scriptRunning;
                        if (scriptRunning)
                        {
                            scriptIP = (selectedLine >= 0) ? selectedLine : 0; /* resume from cursor; cap at 0 for run */
                            scriptActive = 1;                                  /* count as running until stop */
                        }
                        /* on pause: keep cursor where it is, never set selectedLine from scriptIP */
                    }
                    else if (hit == 2)
                    {
                        scriptActive = 0; /* no longer running; allow token/clear again */
                        scriptResetExecution();
                        elapsedTimeSeconds = 0.0f;
                        scriptScrollOffset = 0;
                        selectedLine = 0;
                    }
                    else if (hit == 3)
                    {
                        if (!scriptRunning)
                        {
                            stepRequest = 1;
                            stepRepeatCounter = SCROLL_REPEAT_DELAY;
                        }
                    }
                    else if (hit == 4)
                    {
                        if (scriptRunning)
                        {
                            if (scriptScrollOffset > 0)
                                scriptScrollOffset--;
                        }
                        else if (selectedLine > -1)
                            selectedLine--; /* can go to -1 to insert above first line */
                        scrollUpCounter = SCROLL_REPEAT_DELAY;
                        scrollDownCounter = -1;
                    }
                    else if (hit == 5)
                    {
                        if (scriptRunning)
                        {
                            if (scriptScrollOffset + SCRIPT_LINES_VIS < scriptLines)
                                scriptScrollOffset++;
                        }
                        else if (selectedLine < scriptLines - 1)
                            selectedLine++; /* down from -1 goes to 0 */
                        scrollDownCounter = SCROLL_REPEAT_DELAY;
                        scrollUpCounter = -1;
                    }
                    else if (hit == 6 && !scriptActive)
                    {
                        tokenViewActive = 1;
                    }
                    else if (hit == 7 && !scriptActive)
                    {
                        scriptClearLine();
                        clearRepeatCounter = SCROLL_REPEAT_DELAY;
                    }
                    else if (hit == 8)
                    {
                        scriptSaveToSlot(currentScriptSlot); /* update slot stats before showing menu */
                        scriptMenuActive = 1;
                    }
                }
            }
        }

        /* Physical Start button = Play/Pause (same as tapping the play/pause button) */
        if (kDown & KEY_START && !registerSelectActive && !numpadActive && !tokenViewActive && !scriptMenuActive)
        {
            scriptRunning = !scriptRunning;
            if (scriptRunning)
            {
                scriptIP = (selectedLine >= 0) ? selectedLine : 0; /* cap at 0 for run */
                scriptActive = 1;                                  /* count as running until stop */
            }
            /* on pause: keep cursor where it is */
        }

        /* D-pad Up/Down: only when paused move selected line (when running, use on-screen Up/Down nav to scroll). Up from 0 goes to -1 (insert above first). */
        if (!scriptRunning)
        {
            if (kDown & KEY_UP && selectedLine > -1)
            {
                selectedLine--;
                scrollUpCounter = SCROLL_REPEAT_DELAY;
                scrollDownCounter = -1;
            }
            if (kDown & KEY_DOWN && selectedLine < scriptLines - 1)
            {
                selectedLine++;
                scrollDownCounter = SCROLL_REPEAT_DELAY;
                scrollUpCounter = -1;
            }
        }

        /* Scroll repeat: when holding up/down (touch or D-pad when paused), repeat after delay. Same for Step when paused. */
        if (!registerSelectActive && !numpadActive && !tokenViewActive && !scriptMenuActive)
        {
            int touchOnNav = (keysHeld() & KEY_TOUCH) && touch.px > 0;
            int navHit = touchOnNav ? hitTest(touch.px, touch.py, 0) : 0; /* 0 = held touch, never set selectedLine */
            int scrollUpHeld = (navHit == 4) || (!scriptRunning && (keysHeld() & KEY_UP));
            int scrollDownHeld = (navHit == 5) || (!scriptRunning && (keysHeld() & KEY_DOWN));
            int stepHeld = !scriptRunning && (navHit == 3);
            int clearHeld = (navHit == 7);

            if (!scrollUpHeld)
                scrollUpCounter = -1;
            else if (scrollUpCounter >= 0)
            {
                scrollUpCounter--;
                if (scrollUpCounter == 0)
                {
                    if (scriptRunning)
                    {
                        if (scriptScrollOffset > 0)
                            scriptScrollOffset--;
                    }
                    else if (selectedLine > -1)
                        selectedLine--;
                    scrollUpCounter = SCROLL_REPEAT_INTERVAL;
                }
            }

            if (!scrollDownHeld)
                scrollDownCounter = -1;
            else if (scrollDownCounter >= 0)
            {
                scrollDownCounter--;
                if (scrollDownCounter == 0)
                {
                    if (scriptRunning)
                    {
                        if (scriptScrollOffset + SCRIPT_LINES_VIS < scriptLines)
                            scriptScrollOffset++;
                    }
                    else if (selectedLine < scriptLines - 1)
                        selectedLine++;
                    scrollDownCounter = SCROLL_REPEAT_INTERVAL;
                }
            }

            if (!stepHeld)
                stepRepeatCounter = -1;
            else if (stepRepeatCounter >= 0)
            {
                stepRepeatCounter--;
                if (stepRepeatCounter == 0)
                {
                    scriptStep(1);
                    selectedLine = scriptIP; /* step: align cursor to next token */
                    stepRepeatCounter = SCROLL_REPEAT_INTERVAL;
                }
            }

            if (!clearHeld || scriptActive)
                clearRepeatCounter = -1;
            else if (clearRepeatCounter >= 0)
            {
                clearRepeatCounter--;
                if (clearRepeatCounter == 0)
                {
                    scriptClearLine();
                    clearRepeatCounter = SCROLL_REPEAT_INTERVAL;
                }
            }
        }
        else
        {
            scrollUpCounter = -1;
            scrollDownCounter = -1;
            stepRepeatCounter = -1;
            clearRepeatCounter = -1;
        }

        /* Beep: stop after ~0.1s */
        if (beepFramesLeft > 0)
        {
            beepFramesLeft--;
            if (beepFramesLeft == 0)
                soundPause(beepChannel);
        }

        /* Run script: when running, step until we hit a SLEEP (one loop iteration per frame so ball moves); when sleeping, one step per frame to decrement */
        if (scriptRunning)
        {
            elapsedTimeSeconds += 1.0f / 60.0f;
            if (sleepFramesLeft > 0)
                scriptStep(0); /* one call per frame to decrement sleep */
            else
            {
                int steps = 0;
                while (sleepFramesLeft == 0 && steps < 200)
                {
                    scriptStep(0);
                    steps++;
                    if (scriptIP >= scriptLines)
                    {
                        scriptRunning = 0;
                        break;
                    }
                }
            }
        }
        else if (stepRequest)
        {
            stepRequest = 0;
            scriptStep(1);
            selectedLine = scriptIP; /* step: align cursor to next token */
        }

        /* Redraw bottom screen: register selector, numpad, token view, script menu, or scripting editor */
        if (registerSelectActive)
            drawRegisterSelectScreen();
        else if (numpadActive)
            drawNumpadScreen();
        else if (scriptMenuActive)
            drawScriptMenuScreen();
        else if (tokenViewActive)
            drawTokenViewScreen();
        else
            drawSubScreen();

        /* ----- 3D: clear and draw all active models on top screen ----- */
        {
            /* BACKGROUND index: 0=black, 1=dark pale red, 2=dark pale green, 3=dark pale blue (0-31) */
            int r = 0, g = 0, b = 0;
            if (bgColorIndex == 1)
            {
                r = 12;
                g = 4;
                b = 4;
            }
            else if (bgColorIndex == 2)
            {
                r = 4;
                g = 12;
                b = 4;
            }
            else if (bgColorIndex == 3)
            {
                r = 4;
                g = 4;
                b = 12;
            }
            glClearColor(r, g, b, 31);
        }
        glClearDepth(GL_MAX_DEPTH);
        glClearPolyID(63);

        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();
        {
            float yaw_rad = camYaw * (3.14159265f / 180.0f);
            float pitch_rad = camPitch * (3.14159265f / 180.0f);
            float dx = cosf(pitch_rad) * sinf(yaw_rad);
            float dy = -sinf(pitch_rad);
            float dz = -cosf(pitch_rad) * cosf(yaw_rad);
            gluLookAt(camX, camY, camZ, camX + dx, camY + dy, camZ + dz, 0, 1, 0);
        }
        {
            int i;
            for (i = 0; i < MAX_MODELS; i++)
            {
                if (!modelActive[i])
                    continue;
                glPushMatrix();
                glTranslatef(modelX[i], modelY[i], modelZ[i]);
                glRotatef(modelAngle[i], 0, 1, 0);
                drawCube(CUBE_COLORS[modelColorIndex[i]]);
                drawWireframeCube();
                glPopMatrix(1);
            }
        }

        glFlush(0);
    }

    return 0;
}
